﻿//----------------------------------------------------------------------------
// File : asdxMath.h
// Desc : Math Module.
// Copyright(c) Project Asura. All right reserved.
//----------------------------------------------------------------------------

#ifndef __ASDX_MATH_H__
#define __ASDX_MATH_H__

//----------------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------------
#include <asdxTypedef.h>
#include <cmath>
#include <cfloat>
#include <cassert>
#include <cstring>


namespace asdx {

//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------
struct Vector2;
struct Vector3;
struct Vector4;
struct Matrix;
struct Quaternion;


//----------------------------------------------------------------------------
// Constant Variables
//----------------------------------------------------------------------------
const f32     F_PI        = 3.1415926535897932384626433832795f;     //!< πです.
const f32     F_2PI       = 6.283185307179586476925286766559f;      //!< 2πです.
const f32     F_1DIVPI    = 0.31830988618379067153776752674503f;    //!< 1/πです.
const f32     F_PIDIV2    = 1.5707963267948966192313216916398f;     //!< π/2です.
const f32     F_PIDIV3    = 1.0471975511965977461542144610932f;     //!< π/3です.
const f32     F_PIDIV4    = 0.78539816339744830961566084581988f;    //!< π/4です.
const f32     F_PIDIV6    = 0.52359877559829887307710723054658f;    //!< π/6です.
const f32     F_EPSILON   = 1.192092896e-07f;
const f64     D_EPSILON   = 2.2204460492503131e-016;


//----------------------------------------------------------------------------
// Function
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
//! @brief      度をラジアンに変換します.
//!
//! @param [in]     degree      角度(度)
//! @return     度をラジアンに変換した結果を返却します.
//----------------------------------------------------------------------------
f32     ToRadian( f32 degree );

//-----------------------------------------------------------------------------
//! @brief      ラジアンを度に変換します.
//!
//! @param [in]     radian      角度(ラジアン)
//! @return     ラジアンを度に変換した結果を返却します.
//-----------------------------------------------------------------------------
f32     ToDegree( f32 radian );

//-----------------------------------------------------------------------------
//! @brief      値がゼロであるかどうか判定します.
//!
//! @param [in]     value       判定する値.
//! @return     値がゼロであるとみなせる場合にtrueを返却します.
//-----------------------------------------------------------------------------
bool    IsZero( f32 value );

//-----------------------------------------------------------------------------
//! @brief      値がゼロであるかどうか判定します.
//!
//! @param [in]     value       判定する値.
//! @return     値がゼロであるとみなせる場合にtrueを返却します.
//-----------------------------------------------------------------------------
bool    IsZero( f64 value );

//-----------------------------------------------------------------------------
//! @brief      値が等価であるか判定します.
//!
//! @param [in]     a           判定する値.
//! @param [in]     b           判定する値.
//! @return     値が等価であるとみなせる場合にtrueを返却します.
//-----------------------------------------------------------------------------
bool    IsEqual( f32 a, f32 b );

//-----------------------------------------------------------------------------
//! @brief      値が等価であるか判定します.
//!
//! @param [in]     a           判定する値.
//! @param [in]     b           判定する値.
//! @return     値が等価であるとみなせる場合にtrueを返却します.
//-----------------------------------------------------------------------------
bool    IsEqual( f64 a, f64 b );

//------------------------------------------------------------------------------
//! @brief      非数であるか判定します.
//!
//! @param [in]     value       判定する値.
//! @return     非数であった場合にtrueを返却します.
//------------------------------------------------------------------------------
bool    IsNan( f32 value );

//------------------------------------------------------------------------------
//! @brief      無限大であるか判定します.
//!
//! @param [in]     value       判定する値.
//! @return     無限大であった場合にtrueを返却します.
//------------------------------------------------------------------------------
bool    IsInf( f32 value );

//------------------------------------------------------------------------------
//! @brief      階乗を計算します.
//!
//! @param [in]     number      階乗を計算する値.
//! @return     (number)!を計算した値を返却します.
//------------------------------------------------------------------------------
u32     Fact( u32 number );

//-------------------------------------------------------------------------------
//! @brief      2重階乗を計算します.
//!
//! @param [in]     number      2重階乗を計算する値.
//! @return     (number)!!を計算した値を返却します.
//-------------------------------------------------------------------------------
u32     DblFact( u32 number );

//-------------------------------------------------------------------------------
//! @brief      順列を計算します.
//!
//! @param [in]     n       総数.
//! @param [in]     r       選択数.
//! @return     n個のものからr個とった順列を返却します.
//-------------------------------------------------------------------------------
u32     Perm( u32 n, u32 r );

//-------------------------------------------------------------------------------
//! @brief      組合せを計算します.
//!
//! @param [in]     n       総数.
//! @param [in]     r       選択数.
//! @return     n個のものからr個とった組合せを返却します.
//-------------------------------------------------------------------------------
u32     Comb( u32 n, u32 r );

//-------------------------------------------------------------------------------
//! @brief      フレネル項を計算します.
//!
//! @param [in]     n1          入射側の屈折率です.
//! @param [in]     n2          出射側の屈折率です.
//! @param [in]     cosTheta    cosθの値です
//! @return     Schilickによる近似式を計算した結果を返却します.
//-------------------------------------------------------------------------------
f32     Fresnel( f32 n1, f32 n2, f32 cosTheta );

//-------------------------------------------------------------------------------
//! @brief      フレネル項を計算します.
//!
//! @param [in]     n1          入射側の屈折率です.
//! @param [in]     n2          出射側の屈折率です.
//! @param [in]     cosTheta    cosθの値です
//! @return     Schilickによる近似式を計算した結果を返却します.
//-------------------------------------------------------------------------------
f64     Fresnel( f64 n1, f64 n2, f64 cosTheta );

//------------------------------------------------------------------------------
//! @brief      f32型からf16型に変換します.
//!
//! @param [in]     value       f16型に変換する値.
//! @return     半精度浮動小数表現に変換した結果を返却します.
//------------------------------------------------------------------------------
f16     F32ToF16( f32 value );

//------------------------------------------------------------------------------
//! @brief      f16型からf32型に変換します.
//!
//! @param [in]     value       f32型に変換する値.
//! @return     単精度浮動小数表現に変換した結果を返却します.
//------------------------------------------------------------------------------
f32     F16ToF32( f16 value );

//------------------------------------------------------------------------------
//! @brief      2つの値のうち，大きい方を返却します.
//!
//! @param [in]     a       判定する値.
//! @param [in]     b       判定する値.
//! @return     2つの値のうち，大きい方を返却します.
//------------------------------------------------------------------------------
ASDX_TEMPLATE_INLINE(T)
T Max( T a, T b )
{ return ( a > b ) ? a : b; }

//------------------------------------------------------------------------------
//! @brief      2つの値のうち，小さい方の値を返却します.
//!
//! @param [in]     a       判定する値.
//! @param [in]     b       判定する値.
//! @return     2つの値のうち，小さい方の値を返却します.
//------------------------------------------------------------------------------
ASDX_TEMPLATE_INLINE(T)
T Min( T a, T b )
{ return ( a < b ) ? a : b; }

//------------------------------------------------------------------------------
//! @brief      値を指定された範囲内に収めます.
//!
//! @param [in]     value   クランプする値.
//! @param [in]     a       最小値.
//! @param [in]     b       最大値.
//! @return     値をaからbの範囲内に収めた結果を返却します.
//------------------------------------------------------------------------------
ASDX_TEMPLATE_INLINE(T)
T Clamp( T value, T mini, T maxi )
{ return Max( mini, Min( maxi, value ) ); }

//------------------------------------------------------------------------------
//! @brief      値を0～1に収めます.
//!
//! @param [in]     value   クランプする値.
//! @return     値を0から1の範囲内に収めた結果を返却します.
//------------------------------------------------------------------------------
ASDX_TEMPLATE_INLINE(T)
T Saturate( T value )
{ return Clamp( value, T(0), T(1) ); }

//------------------------------------------------------------------------------
//! @brief      指定された値の符号を返却します.
//!
//! @param [in]     value   符号を取得する値.
//! @return     符号が正である場合には1を，負である場合には-1を返却します.
//------------------------------------------------------------------------------
ASDX_TEMPLATE_INLINE(T)
T Sign( T value )
{ return ( value < T(0) ) ? T(-1) : T(1); }


////////////////////////////////////////////////////////////////////////////////
// Vector2 strucutre
////////////////////////////////////////////////////////////////////////////////
typedef struct Vector2
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @param [in]     value       乗算されるベクトル.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    friend Vector2   operator*   ( f32, const Vector2& );

public:
    //==========================================================================
    // public variables
    //==========================================================================
    f32 x;  //!< X成分です.
    f32 y;  //!< Y成分です.

    //==========================================================================
    // public methods
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Vector2();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValeus     要素数2の配列.
    //--------------------------------------------------------------------------
    Vector2( const f32* );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nx           X成分.
    //! @param [in]     ny           Y成分.
    //--------------------------------------------------------------------------
    Vector2( const f32 nx, const f32 ny );

    //--------------------------------------------------------------------------
    //! @brief      f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator       f32* ();

    //--------------------------------------------------------------------------
    //! @brief      const f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      加算代入演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2&         operator += ( const Vector2& );

    //--------------------------------------------------------------------------
    //! @brief      減算代入演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2&         operator -= ( const Vector2& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2&         operator *= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2&         operator /= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Vector2&         operator =  ( const Vector2& );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身の値を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator +  () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //!
    //! @return     負符号を付けた値を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator -  () const;

    //--------------------------------------------------------------------------
    //! @brief      加算演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator +  ( const Vector2& ) const;

    //--------------------------------------------------------------------------
    //! @brief      減算演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator -  ( const Vector2& ) const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator *  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector2          operator /  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator == ( const Vector2& ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が非等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator != ( const Vector2& ) const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さを求めます.
    //!
    //! @return     ベクトルの長さを返却します.
    //--------------------------------------------------------------------------
    f32             Length          () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さの2乗値を求めます.
    //!
    //! @return     ベクトルの長さの2乗値を返却します.
    //--------------------------------------------------------------------------
    f32             LengthSq        () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector2&        Normalize       ();

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       長さが0の場合に設定する値.
    //! @return     ベクトルの長さが0の場合はvalueを，そうでないときは正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector2&        SafeNormalize   ( const Vector2& );


    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求める値.
    //! @return     各成分の絶対値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Abs( const Vector2& value );

    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求めます.
    //! @param [out]    result      計算結果の格納先.
    //--------------------------------------------------------------------------
    static void    Abs( const Vector2& value, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @return     クランプされた値を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Clamp( const Vector2& value, const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Clamp( const Vector2& value, const Vector2& a, const Vector2& b, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @return     クランプされた値.
    //--------------------------------------------------------------------------
    static Vector2 Saturate( const Vector2& value );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Saturate( const Vector2& value, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離を返却します.
    //--------------------------------------------------------------------------
    static f32     Distance( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離.
    //--------------------------------------------------------------------------
    static void    Distance( const Vector2& a, const Vector2& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離の2乗値を返却します.
    //--------------------------------------------------------------------------
    static f32     DistanceSq( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離の2乗値.
    //--------------------------------------------------------------------------
    static void    DistanceSq( const Vector2& a, const Vector2& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     ベクトルの内積を返却します.
    //--------------------------------------------------------------------------
    static f32     Dot( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      ベクトルの内積.
    //--------------------------------------------------------------------------
    static void    Dot( const Vector2& a, const Vector2& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector2 Normalize( const Vector2& value );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [out]    result      正規化したベクトル.
    //--------------------------------------------------------------------------
    static void    Normalize( const Vector2& value, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @return     ベクトルの長さが0でなければ正規化したベクトルを返却,
    //!             長さが0の場合はsetを返却します.
    //--------------------------------------------------------------------------
    static Vector2 SafeNormalize( const Vector2& value, const Vector2& set );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @param [out]    result      長さが0でなければ正規化したベクトル，0であれば set.
    //--------------------------------------------------------------------------
    static void    SafeNormalize( const Vector2& value, const Vector2& set, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトルの交差角をラジアンで返却します.
    //--------------------------------------------------------------------------
    static f32     ComputeCrossingAngle( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトルの交差角(ラジアン).
    //--------------------------------------------------------------------------
    static void    ComputeCrossingAngle( const Vector2& a, const Vector2& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最小値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Min( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最小値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Min( const Vector2& a, const Vector2& b, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最大値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Max( const Vector2& a, const Vector2& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最大値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Max( const Vector2& a, const Vector2& b, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルから，反射ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @return     反射ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector2 Reflect( const Vector2& i, const Vector2& n );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルから，反射ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [out]    result      反射ベクトル.
    //--------------------------------------------------------------------------
    static void    Reflect( const Vector2& i, const Vector2& n, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルと屈折角から，屈折ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [in]     eta         屈折率.
    //! @return     屈折ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector2 Refract( const Vector2& i, const Vector2& n, const f32 eta );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルと屈折角から，屈折ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [in]     eta         屈折率.
    //! @param [out]    result      屈折ベクトル.
    //--------------------------------------------------------------------------
    static void    Refract( const Vector2& i, const Vector2& n, const f32 eta, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @return     重心座標上の点を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Barycentric( const Vector2& a, const Vector2& b, const Vector2& c, const f32 f, const f32 g );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @param [out]    result      重心座標上の点.
    //--------------------------------------------------------------------------
    static void    Barycentric( const Vector2& a, const Vector2& b, const Vector2& c, const f32 f, const f32 g, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @return     エルミートスプライン補間を行った結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Hermite( const Vector2& a, const Vector2& t1, const Vector2& b, const Vector2& t2, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      エルミートスプライン補間の結果.
    //--------------------------------------------------------------------------
    static void    Hermite( const Vector2& a, const Vector2& t1, const Vector2& b, const Vector2& t2, const f32 amount, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @return     Catmull-Rom補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 CatmullRom( const Vector2& a, const Vector2& b, const Vector2& c, const Vector2& d, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @param [out]    result      Catmull-Rom補間の結果.
    //--------------------------------------------------------------------------
    static void    CatmullRom( const Vector2& a, const Vector2& b, const Vector2& c, const Vector2& d, const f32 amount, Vector2& result );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @return     線形補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 Lerp( const Vector2& a, const Vector2& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @param [out]    result      線形補間の結果.
    //--------------------------------------------------------------------------
    static void    Lerp( const Vector2& a, const Vector2& b, const f32 amount, Vector2 &result );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @return     補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector2 SmoothStep( const Vector2& a, const Vector2& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      補間の結果.
    //--------------------------------------------------------------------------
    static void    SmoothStep( const Vector2& a, const Vector2& b, const f32 amount, Vector2 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     変換されたベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector2 Transform( const Vector2& position, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      変換されたベクトル.
    //--------------------------------------------------------------------------
    static void    Transform( const Vector2& position, const Matrix& matrix, Vector2 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，法線ベクトルを変換します.
    //!
    //! @param [in]     normal      入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     変換された法線ベクトル.
    //--------------------------------------------------------------------------
    static Vector2 TransformNormal( const Vector2& normal, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，法線ベクトルを変換します.
    //!
    //! @param [in]     normal      入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      変換された法線ベクトル.
    //--------------------------------------------------------------------------
    static void    TransformNormal( const Vector2& normal, const Matrix& matrix, Vector2 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
    //!
    //! @param [in]     coord       入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     行列変換後, w=1に射影されたベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector2 TransformCoord( const Vector2& coords, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
    //!
    //! @param [in]     coord       入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      行列変換後，w=1に射影されたベクトル.
    //--------------------------------------------------------------------------
    static void    TransformCoord( const Vector2& coords, const Matrix& matrix, Vector2 &result );

} Vector2;


////////////////////////////////////////////////////////////////////////////////
// Vector3 structure
////////////////////////////////////////////////////////////////////////////////
typedef struct Vector3
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @param [in]     value       乗算されるベクトル.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    friend Vector3   operator *  ( f32, const Vector3& );

public:
    //==========================================================================
    // public variables
    //==========================================================================
    f32 x;  //!< X成分です.
    f32 y;  //!< Y成分です.
    f32 z;  //!< Z成分です.

    //==========================================================================
    // public methods
    //==========================================================================
    
    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Vector3();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValeus     要素数3の配列.
    //--------------------------------------------------------------------------
    Vector3( const f32 * );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     value       2次元ベクトル.
    //! @param [in]     nz          Z成分.
    //--------------------------------------------------------------------------
    Vector3( const Vector2& value, const f32 nz );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nx           X成分.
    //! @param [in]     ny           Y成分.
    //! @param [in]     nz           Z成分.
    //--------------------------------------------------------------------------
    Vector3( const f32 nx, const f32 ny, const f32 nz );

    //--------------------------------------------------------------------------
    //! @brief      f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator       f32* ();

    //--------------------------------------------------------------------------
    //! @brief      const f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      加算代入演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3&         operator += ( const Vector3& );

    //--------------------------------------------------------------------------
    //! @brief      減算代入演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3&         operator -= ( const Vector3& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3&         operator *= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3&         operator /= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Vector3&         operator =  ( const Vector3& );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身の値を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator +  () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //!
    //! @return     負符号を付けた値を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator -  () const;

    //--------------------------------------------------------------------------
    //! @brief      加算演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator +  ( const Vector3& ) const;

    //--------------------------------------------------------------------------
    //! @brief      減算演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator -  ( const Vector3& ) const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator *  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector3          operator /  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator == ( const Vector3& ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が非等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator != ( const Vector3& ) const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さを求めます.
    //!
    //! @return     ベクトルの長さを返却します.
    //--------------------------------------------------------------------------
    f32             Length          () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さの2乗値を求めます.
    //!
    //! @return     ベクトルの長さの2乗値を返却します.
    //--------------------------------------------------------------------------
    f32             LengthSq        () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector3&        Normalize       ();

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       長さが0の場合に設定する値.
    //! @return     ベクトルの長さが0の場合はvalueを，そうでないときは正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector3&        SafeNormalize   ( const Vector3& );


    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求める値.
    //! @return     各成分の絶対値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Abs( const Vector3& value );

    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求めます.
    //! @param [out]    result      計算結果の格納先.
    //--------------------------------------------------------------------------
    static void    Abs( const Vector3& value, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @return     クランプされた値を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Clamp( const Vector3& value, const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Clamp( const Vector3& value, const Vector3& a, const Vector3& b, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @return     クランプされた値.
    //--------------------------------------------------------------------------
    static Vector3 Saturate( const Vector3& value );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Saturate( const Vector3& value, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離を返却します.
    //--------------------------------------------------------------------------
    static f32     Distance( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離.
    //--------------------------------------------------------------------------
    static void    Distance( const Vector3& a, const Vector3& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離の2乗値を返却します.
    //--------------------------------------------------------------------------
    static f32     DistanceSq( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離の2乗値.
    //--------------------------------------------------------------------------
    static void    DistanceSq( const Vector3& a, const Vector3& b, f32& reuslt );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     ベクトルの内積を返却します.
    //--------------------------------------------------------------------------
    static f32     Dot( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      ベクトルの内積.
    //--------------------------------------------------------------------------
    static void    Dot( const Vector3& a, const Vector3& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの外積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     ベクトルの外積を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Cross( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの外積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      ベクトルの外積.
    //--------------------------------------------------------------------------
    static void    Cross( const Vector3& a, const Vector3& b, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 Normalize( const Vector3& value );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [out]    result      正規化したベクトル.
    //--------------------------------------------------------------------------
    static void    Normalize( const Vector3& value, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @return     ベクトルの長さが0でなければ正規化したベクトルを返却,
    //!             長さが0の場合はsetを返却します.
    //--------------------------------------------------------------------------
    static Vector3  SafeNormalize( const Vector3& value, const Vector3& set );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @param [out]    result      長さが0でなければ正規化したベクトル，0であれば set.
    //--------------------------------------------------------------------------
    static void     SafeNormalize( const Vector3& value, const Vector3& set, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      三角形の法線ベクトルを求めます.
    //!
    //! @param [in]     p1          三角形を構成する頂点.
    //! @param [in]     p2          三角形を構成する頂点.
    //! @param [in]     p3          三角形を構成する頂点.
    //! @return     法線ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 ComputeNormal( const Vector3& p1, const Vector3& p2, const Vector3& p3 );

    //--------------------------------------------------------------------------
    //! @brief      三角形の法線ベクトルを求めます.
    //!
    //! @param [in]     p1          三角形を構成する頂点.
    //! @param [in]     p2          三角形を構成する頂点.
    //! @param [in]     p3          三角形を構成する頂点.
    //! @param [out]    result      法線ベクトル.
    //--------------------------------------------------------------------------
    static void    ComputeNormal( const Vector3& p1, const Vector3& p2, const Vector3& p3, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      四角形の法線ベクトルを求めます.
    //!
    //! @param [in]     p1          四角形を構成する頂点.
    //! @param [in]     p2          四角形を構成する頂点.
    //! @param [in]     p3          四角形を構成する頂点.
    //! @param [in]     p4          四角形を構成する頂点.
    //! @return     法線ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 ComputeQuadNormal( const Vector3& p1, const Vector3& p2, const Vector3& p3, const Vector3& p4 );

    //--------------------------------------------------------------------------
    //! @brief      四角形の法線ベクトルを求めます.
    //!
    //! @param [in]     p1          四角形を構成する頂点.
    //! @param [in]     p2          四角形を構成する頂点.
    //! @param [in]     p3          四角形を構成する頂点.
    //! @param [in]     p4          四角形を構成する頂点.
    //! @param [out]    result      法線ベクトル.
    //--------------------------------------------------------------------------
    static void    ComputeQuadNormal( const Vector3& p1, const Vector3& p2, const Vector3& p3, const Vector3& p4, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトルの交差角(ラジアン).
    //--------------------------------------------------------------------------
    static f32     ComputeCrossingAngle( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトルの交差角(ラジアン).
    //--------------------------------------------------------------------------
    static void    ComputeCrossingAngle( const Vector3& a, const Vector3& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最小値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Min( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最小値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Min( const Vector3& a, const Vector3& b, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最大値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Max( const Vector3& a, const Vector3& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最大値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Max( const Vector3& a, const Vector3& b, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルから，反射ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @return     反射ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 Reflect( const Vector3& i, const Vector3& n );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルから，反射ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [out]    result      反射ベクトル.
    //--------------------------------------------------------------------------
    static void    Reflect( const Vector3& i, const Vector3& n, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルと屈折角から，屈折ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [in]     eta         屈折率.
    //! @return     屈折ベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 Refract( const Vector3& i, const Vector3& n, const f32 eta );

    //--------------------------------------------------------------------------
    //! @brief      指定された法線を持つ表面の入射ベクトルと屈折角から，屈折ベクトルを求めます.
    //!
    //! @param [in]     i           入射ベクトル.
    //! @param [in]     n           法線ベクトル.
    //! @param [in]     eta         屈折率.
    //! @param [out]    result      屈折ベクトル.
    //--------------------------------------------------------------------------
    static void    Refract( const Vector3& i, const Vector3& n, const f32 eta, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @return     重心座標上の点を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Barycentric( const Vector3& a, const Vector3& b, const Vector3& v3, const f32 f, const f32 g );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @param [out]    result      重心座標上の点.
    //--------------------------------------------------------------------------
    static void    Barycentric( const Vector3& a, const Vector3& b, const Vector3& v3, const f32 f, const f32 g, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @return     エルミートスプライン補間を行った結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Hermite( const Vector3& a, const Vector3& t1, const Vector3& b, const Vector3& t2, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      エルミートスプライン補間の結果.
    //--------------------------------------------------------------------------
    static void    Hermite( const Vector3& a, const Vector3& t1, const Vector3& b, const Vector3& t2, const f32 amount, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @return     Catmull-Rom補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 CatmullRom( const Vector3& a, const Vector3& b, const Vector3& c, const Vector3& d, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @param [out]    result      Catmull-Rom補間の結果.
    //--------------------------------------------------------------------------
    static void    CatmullRom( const Vector3& a, const Vector3& b, const Vector3& c, const Vector3& d, const f32 amount, Vector3& result );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @return     線形補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 Lerp( const Vector3& a, const Vector3& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @param [out]    result      線形補間の結果.
    //--------------------------------------------------------------------------
    static void    Lerp( const Vector3& a, const Vector3& b, const f32 amount, Vector3 &result );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @return     補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector3 SmoothStep( const Vector3& a, const Vector3& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      補間の結果.
    //--------------------------------------------------------------------------
    static void    SmoothStep( const Vector3& a, const Vector3& b, const f32 amount, Vector3 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     変換されたベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 Transform( const Vector3& position, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      変換されたベクトル.
    //--------------------------------------------------------------------------
    static void    Transform( const Vector3& position, const Matrix& matrix, Vector3 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，法線ベクトルを変換します.
    //!
    //! @param [in]     normal      入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     変換された法線ベクトル.
    //---------------------------------------------------------------------------
    static Vector3 TransformNormal( const Vector3& normal, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，法線ベクトルを変換します.
    //!
    //! @param [in]     normal      入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      変換された法線ベクトル.
    //--------------------------------------------------------------------------
    static void    TransformNormal( const Vector3& normal, const Matrix& matrix, Vector3 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
    //!
    //! @param [in]     coord       入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     行列変換後, w=1に射影されたベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector3 TransformCoord( const Vector3& coord, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いてベクトルを変換し，変換結果をw=1に射影します.
    //!
    //! @param [in]     coord       入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      行列変換後，w=1に射影されたベクトル.
    //--------------------------------------------------------------------------
    static void    TransformCoord( const Vector3& coord, const Matrix& matrix, Vector3& result );

    //---------------------------------------------------------------------------
    //! @brief      スカラー3重積を計算します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @return     スカラー3重積の演算結果を返却します.
    //----------------------------------------------------------------------------
    static f32     ScalarTriple( const Vector3& a, const Vector3& b, const Vector3& c );

    //---------------------------------------------------------------------------
    //! @brief      スカラー3重積を計算します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [out]    result      スカラー3重積の演算結果.
    //----------------------------------------------------------------------------
    static void    ScalarTriple( const Vector3& a, const Vector3& b, const Vector3& c, f32& result );

    //---------------------------------------------------------------------------
    //! @brief      ベクトル3重積を計算します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @return     ベクトル3重積の演算結果を返却します.
    //----------------------------------------------------------------------------
    static Vector3 VectorTriple( const Vector3& a, const Vector3& b, const Vector3& c );

    //---------------------------------------------------------------------------
    //! @brief      ベクトル3重積を計算します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [out]    result      ベクトル3重積の演算結果.
    //----------------------------------------------------------------------------
    static void    VectorTriple( const Vector3& a, const Vector3& b, const Vector3& c, Vector3& result );

} Vector3;


/////////////////////////////////////////////////////////////////////////////////
// Vector4 structure
/////////////////////////////////////////////////////////////////////////////////
typedef struct Vector4
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @param [in]     value       乗算されるベクトル.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    friend Vector4   operator *  ( f32, const Vector4& );

public:
    //==========================================================================
    // private variables
    //==========================================================================
    f32 x;      //!< X成分です.
    f32 y;      //!< Y成分です.
    f32 z;      //!< Z成分です.
    f32 w;      //!< W成分です.

    //==========================================================================
    // public methods
    //==========================================================================
    
    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Vector4();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValeus     要素数4の配列.
    //--------------------------------------------------------------------------
    Vector4( const f32* );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     value       2次元ベクトル.
    //! @param [in]     nz          Z成分.
    //! @param [in]     nw          W成分.
    //--------------------------------------------------------------------------
    Vector4( const Vector2& value, const f32 nz, const f32 nw );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     value       3次元ベクトル.
    //! @param [in]     nw          W成分.
    //--------------------------------------------------------------------------
    Vector4( const Vector3& value, const f32 nw );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nx           X成分.
    //! @param [in]     ny           Y成分.
    //! @param [in]     nz           Z成分.
    //! @param [in]     nw           W成分.
    //--------------------------------------------------------------------------
    Vector4( const f32 nx, const f32 ny, const f32 nz, const f32 nw );

    //--------------------------------------------------------------------------
    //! @brief      f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator       f32* ();

    //--------------------------------------------------------------------------
    //! @brief      const f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      加算代入演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4&         operator += ( const Vector4& );

    //--------------------------------------------------------------------------
    //! @brief      減算代入演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4&         operator -= ( const Vector4& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4&         operator *= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4&         operator /= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Vector4&         operator =  ( const Vector4& );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身の値を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator +  () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //!
    //! @return     負符号を付けた値を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator -  () const;

    //--------------------------------------------------------------------------
    //! @brief      加算演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator +  ( const Vector4& ) const;

    //--------------------------------------------------------------------------
    //! @brief      減算演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator -  ( const Vector4& ) const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator *  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Vector4          operator /  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator == ( const Vector4& ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @return     値が非等価であればtrue, そうでなければfalseを返却します.
    //--------------------------------------------------------------------------
    bool             operator != ( const Vector4& ) const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さを求めます.
    //!
    //! @return     ベクトルの長さを返却します.
    //--------------------------------------------------------------------------
    f32              Length         () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの長さの2乗値を求めます.
    //!
    //! @return     ベクトルの長さの2乗値を返却します.
    //--------------------------------------------------------------------------
    f32              LengthSq       () const;

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector4&         Normalize      ();

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       長さが0の場合に設定する値.
    //! @return     ベクトルの長さが0の場合はvalueを，そうでないときは正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector4&         SafeNormalize  ( const Vector4& );


    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求める値.
    //! @return     各成分の絶対値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Abs( const Vector4& v );

    //--------------------------------------------------------------------------
    //! @brief      各成分の絶対値を求めます.
    //!
    //! @param [in]     value       絶対値を求めます.
    //! @param [out]    result      計算結果の格納先.
    //--------------------------------------------------------------------------
    static void    Abs( const Vector4& value, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @return     クランプされた値を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Clamp( const Vector4& value, const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      値を指定された範囲内に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [in]     a           最小値.
    //! @param [in]     b           最大値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Clamp( const Vector4& value, const Vector4& a, const Vector4& b, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @return     クランプされた値.
    //--------------------------------------------------------------------------
    static Vector4 Saturate( const Vector4& value );

    //--------------------------------------------------------------------------
    //! @brief      指定された値を0～1の範囲に制限します.
    //!
    //! @param [in]     value       クランプする値.
    //! @param [out]    result      クランプされた値.
    //--------------------------------------------------------------------------
    static void    Saturate( const Vector4& value, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離を返却します.
    //--------------------------------------------------------------------------
    static f32     Distance( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離.
    //--------------------------------------------------------------------------
    static void    Distance( const Vector4& a, const Vector4& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトル間の距離の2乗値を返却します.
    //--------------------------------------------------------------------------
    static f32     DistanceSq( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトル間の距離の2乗値を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトル間の距離の2乗値.
    //--------------------------------------------------------------------------
    static void    DistanceSq( const Vector4& a, const Vector4& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     ベクトルの内積を返却します.
    //--------------------------------------------------------------------------
    static f32     Dot( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの内積を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      ベクトルの内積.
    //--------------------------------------------------------------------------
    static void    Dot( const Vector4& a, const Vector4& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @return     正規化したベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector4 Normalize( const Vector4& value );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルを正規化します.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [out]    result      正規化したベクトル.
    //--------------------------------------------------------------------------
    static void    Normalize( const Vector4& value, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @return     ベクトルの長さが0でなければ正規化したベクトルを返却,
    //!             長さが0の場合はsetを返却します.
    //--------------------------------------------------------------------------
    static Vector4 SafeNormalize( const Vector4& value, const Vector4& set );

    //--------------------------------------------------------------------------
    //! @brief      ベクトルの正規化を試みます.
    //!
    //! @param [in]     value       正規化するベクトル.
    //! @param [in]     set         長さが0の場合に設定するベクトル.
    //! @param [out]    result      長さが0でなければ正規化したベクトル，0であれば set.
    //--------------------------------------------------------------------------
    static void    SafeNormalize( const Vector4& value, const Vector4& set, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @return     2つのベクトルの交差角をラジアンで返却します.
    //--------------------------------------------------------------------------
    static f32     ComputeCrossingAngle( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      2つのベクトルの交差角を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [out]    result      2つのベクトルの交差角(ラジアン).
    //--------------------------------------------------------------------------
    static void    ComputeCrossingAngle( const Vector4& a, const Vector4& b, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最小値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Min( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最小値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最小値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Min( const Vector4& a, const Vector4& b, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @return     各成分の最大値を求め，その結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Max( const Vector4& a, const Vector4& b );

    //--------------------------------------------------------------------------
    //! @brief      各成分の最大値を求めます.
    //!
    //! @param [in]     a           比較する値.
    //! @param [in]     b           比較する値.
    //! @param [out]    result      各成分の最大値を求めた結果.
    //--------------------------------------------------------------------------
    static void    Max( const Vector4& a, const Vector4& b, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @return     重心座標上の点を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Barycentric( const Vector4& a, const Vector4& b, const Vector4& c, const f32 f, const f32 g );

    //--------------------------------------------------------------------------
    //! @brief      重心座標上の点を求めます.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     c           入力ベクトル.
    //! @param [in]     amount1     重み.
    //! @param [in]     amount2     重み.
    //! @param [out]    result      重心座標上の点.
    //--------------------------------------------------------------------------
    static void    Barycentric( const Vector4& a, const Vector4& b, const Vector4& c, const f32 f, const f32 g, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @return     エルミートスプライン補間を行った結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Hermite( const Vector4& a, const Vector4& t1, const Vector4& b, const Vector4& t2, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      エルミートスプライン補間を行います.
    //!
    //! @param [in]     a           入力位置ベクトル.
    //! @param [in]     t1          入力接ベクトル.
    //! @param [in]     b           入力位置ベクトル.
    //! @param [in]     t2          入力接ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      エルミートスプライン補間の結果.
    //--------------------------------------------------------------------------
    static void    Hermite( const Vector4& a, const Vector4& t1, const Vector4& b, const Vector4& t2, const f32 amount, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @return     Catmull-Rom補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 CatmullRom( const Vector4& a, const Vector4& b, const Vector4& c, const Vector4& d, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      Catmull-Rom補間を行います.
    //!
    //! @param [in]     a           補間の最初の位置.
    //! @param [in]     b           補間の2番目の位置.
    //! @param [in]     c           補間の3番目の位置.
    //! @param [in]     d           補間の4番目の位置.
    //! @param [in]     amount      加重係数.
    //! @param [out]    result      Catmull-Rom補間の結果.
    //--------------------------------------------------------------------------
    static void    CatmullRom( const Vector4& a, const Vector4& b, const Vector4& c, const Vector4& d, const f32 amount, Vector4& result );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @return     線形補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 Lerp( const Vector4& a, const Vector4& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      線形補間を行います.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み(0～1の値範囲で指定).
    //! @param [out]    result      線形補間の結果.
    //--------------------------------------------------------------------------
    static void    Lerp( const Vector4& a, const Vector4& b, const f32 amount, Vector4 &result );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @return     補間の結果を返却します.
    //--------------------------------------------------------------------------
    static Vector4 SmoothStep( const Vector4& a, const Vector4& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      3次方程式を用いて，2つの値の間を補間します.
    //!
    //! @param [in]     a           入力ベクトル.
    //! @param [in]     b           入力ベクトル.
    //! @param [in]     amount      重み.
    //! @param [out]    result      補間の結果.
    //--------------------------------------------------------------------------
    static void    SmoothStep( const Vector4& a, const Vector4& b, const f32 amount, Vector4 &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @return     変換されたベクトルを返却します.
    //--------------------------------------------------------------------------
    static Vector4 Transform( const Vector4& position, const Matrix& matrix );

    //--------------------------------------------------------------------------
    //! @brief      指定された行列を用いて，ベクトルを変換します.
    //!
    //! @param [in]     position    入力ベクトル.
    //! @param [in]     matrix      変換行列.
    //! @param [out]    result      変換されたベクトル.
    //--------------------------------------------------------------------------
    static void    Transform( const Vector4& position, const Matrix& matrix, Vector4 &result );

} Vector4;


////////////////////////////////////////////////////////////////////////////////
// Matrix structure
// 行列クラス    (列優先行列)
////////////////////////////////////////////////////////////////////////////////
typedef struct Matrix 
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @param [in]     value       乗算される行列.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    friend Matrix operator * ( f32, const Matrix& );

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    union
    {
        struct
        {
            f32 _11, _12, _13, _14;
            f32 _21, _22, _23, _24;
            f32 _31, _32, _33, _34;
            f32 _41, _42, _43, _44;
        };
        f32 m[4][4];
    };

    //==========================================================================
    // public methods.
    //==========================================================================
    
    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Matrix();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValues     要素数16の配列.
    //--------------------------------------------------------------------------
    Matrix( const f32* );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     m11         1行1列の値.
    //! @param [in]     m12         1行2列の値.
    //! @param [in]     m13         1行3列の値.
    //! @param [in]     m14         1行4列の値.
    //! @param [in]     m21         2行1列の値.
    //! @param [in]     m22         2行2列の値.
    //! @param [in]     m23         2行3列の値.
    //! @param [in]     m24         2行4列の値.
    //! @param [in]     m31         3行1列の値.
    //! @param [in]     m32         3行2列の値.
    //! @param [in]     m33         3行3列の値.
    //! @param [in]     m34         3行4列の値.
    //! @param [in]     m41         4行1列の値.
    //! @param [in]     m42         4行2列の値.
    //! @param [in]     m43         4行3列の値.
    //! @param [in]     m44         4行4列の値.
    //--------------------------------------------------------------------------
    Matrix ( const f32 m11, const f32 m12, const f32 m13, const f32 m14,
             const f32 m21, const f32 m22, const f32 m23, const f32 m24,
             const f32 m31, const f32 m32, const f32 m33, const f32 m34,
             const f32 m41, const f32 m42, const f32 m43, const f32 m44 );

    //--------------------------------------------------------------------------
    //! @brief      インデクサです.
    //!
    //! @param [in]     row         行番号.
    //! @param [in]     col         列番号.
    //! @return     指定された行番号と列番号に対応する要素を返却します.
    //--------------------------------------------------------------------------
    f32& operator () ( u32 row, u32 col );

    //--------------------------------------------------------------------------
    //! @brief      インデクサです(const版).
    //!
    //! @param [in]     row         行番号.
    //! @param [in]     col         列番号.
    //! @return     指定された行番号と列番号に対応する要素を返却します.
    //--------------------------------------------------------------------------
    f32  operator () ( u32 row, u32 col ) const;

    //--------------------------------------------------------------------------
    //! @brief      f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //-------------------------------------------------------------------------
    operator       f32* ();

    //--------------------------------------------------------------------------
    //! @brief      const f32*型への演算子です.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     value       乗算する行列.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator *= ( const Matrix& );

    //--------------------------------------------------------------------------
    //! @brief      加算代入演算子です.
    //!
    //! @param [in]     value       加算する行列.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator += ( const Matrix& );

    //--------------------------------------------------------------------------
    //! @brief      減算代入演算子です.
    //!
    //! @param [in]     value       減算する行列.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator -= ( const Matrix& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator *= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     除算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator /= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Matrix& operator =  ( const Matrix& );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身を値を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator + () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //
    //! @return     各成分にマイナスを付けた値を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator - () const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     value       乗算する値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator *  ( const Matrix& ) const;

    //--------------------------------------------------------------------------
    //! @brief      加算演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator +  ( const Matrix& ) const;

    //--------------------------------------------------------------------------
    //! @brief      減算演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @retrurn    減算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator -  ( const Matrix& ) const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Matrix  operator *  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //--------------------------------------------------------------------------
    Matrix  operator /  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    値が等価です.
    //! @retval false   値が非等価です.
    //--------------------------------------------------------------------------
    bool    operator == ( const Matrix& ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    値が非等価です.
    //! @retval false   値が等価です.
    //--------------------------------------------------------------------------
    bool    operator != ( const Matrix& ) const;

    //--------------------------------------------------------------------------
    //! @brief      行列式を求めます.
    //!
    //! @return     行列式の値を返却します.
    //--------------------------------------------------------------------------
    f32     Determinant () const;

    //--------------------------------------------------------------------------
    //! @brief      単位行列にします.
    //!
    //! @return     単位行列を返却します.
    //--------------------------------------------------------------------------
    Matrix& Identity    ();


    //--------------------------------------------------------------------------
    //! @brief      単位行列にします.
    //!
    //! @param [in]     value       単位行列にする値.
    //--------------------------------------------------------------------------
    static void    Identity( Matrix &value );

    //--------------------------------------------------------------------------
    //! @brief      単位行列であるか判定します.
    //!
    //! @param [in]     value       判定する値.
    //! @retval true    単位行列です.
    //! @retval false   非単位行列です.
    //--------------------------------------------------------------------------
    static bool    IsIdentity( const Matrix &value );

    //--------------------------------------------------------------------------
    //! @brief      行列を転置します.
    //!
    //! @param [in]     value       転置する行列.
    //! @return     行列を転置した結果を返却します.
    //--------------------------------------------------------------------------
    static Matrix  Transpose( const Matrix& value );

    //--------------------------------------------------------------------------
    //! @brief      行列を転置します.
    //!
    //! @param [in]     value       転置する行列.
    //! @param [out]    result      転置された行列.
    //--------------------------------------------------------------------------
    static void    Transpose( const Matrix& value, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      行列同士を乗算します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    static Matrix  Multiply( const Matrix& a, const Matrix& b );

    //--------------------------------------------------------------------------
    //! @brief      行列同士を乗算します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @param [out]    result      乗算結果.
    //--------------------------------------------------------------------------
    static void    Multiply( const Matrix& a, const Matrix& b, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      スカラー乗算します.
    //!
    //! @param [in]     value       入力行列.
    //! @param [in]     scalar      スカラー値.
    //! @return     行列をスカラー倍した結果を返却します.
    //--------------------------------------------------------------------------
    static Matrix  Multiply( const Matrix& value, const f32 scalar );

    //--------------------------------------------------------------------------
    //! @brief      スカラー乗算します.
    //!
    //! @param [in]     value       入力行列.
    //! @param [in]     scalar      スカラー値.
    //! @param [out]    result      スカラー乗算した結果.
    //--------------------------------------------------------------------------
    static void    Multiply( const Matrix& value, const f32 scalar, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      行列同士を乗算し，乗算結果を転置します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @return     行列同士を乗算し，乗算結果を転置した値を返却します.
    //--------------------------------------------------------------------------
    static Matrix  MultiplyTranspose( const Matrix& a, const Matrix& b );

    //--------------------------------------------------------------------------
    //! @brief      行列同士を乗算し，乗算結果を転置します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @param [out]    result      行列同士を乗算し，乗算結果を転置した値.
    //--------------------------------------------------------------------------
    static void    MultiplyTranspose( const Matrix& a, const Matrix& b, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      逆行列を求めます.
    //!
    //! @param [in]     value       逆行列を求める値.
    //--------------------------------------------------------------------------
    static Matrix  Invert( const Matrix& value );

    //--------------------------------------------------------------------------
    //! @brief      逆行列を求めます.
    //!
    //! @param [in]     value       逆行列を求める値.
    //! @param [out]    result      逆行列.
    //--------------------------------------------------------------------------
    static void    Invert( const Matrix& value, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     scale      拡大縮小値.
    //! @return     拡大縮小行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateScale( const f32 scale );

    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     scale       拡大縮小値.
    //! @param [out]    result      拡大縮小行列.
    //--------------------------------------------------------------------------
    static void    CreateScale( const f32 scale, Matrix &result );
    
    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     sx          X成分の拡大縮小値.
    //! @param [in]     sy          Y成分の拡大縮小値.
    //! @param [in]     sz          Z成分の拡大縮小値.
    //! @return     拡大縮小行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateScale( const f32 sx, const f32 sy, const f32 sz );

    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     sx          X成分の拡大縮小値.
    //! @param [in]     sy          Y成分の拡大縮小値.
    //! @param [in]     sz          Z成分の拡大縮小値.
    //! @param [out]    result      拡大縮小行列.
    //--------------------------------------------------------------------------
    static void    CreateScale( const f32 sx, const f32 sy, const f32 sz, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     scale       拡大縮小値.
    //! @return     拡大縮小行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateScale( const Vector3& scale );

    //--------------------------------------------------------------------------
    //! @brief      拡大縮小行列を生成します.
    //!
    //! @param [in]     scale       拡大縮小値.
    //! @param [out]    result      拡大縮小行列.
    //--------------------------------------------------------------------------
    static void    CreateScale( const Vector3& scale, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      平行移動行列を生成します.
    //!
    //! @param [in]     tx          X成分の平行移動値.
    //! @param [in]     ty          Y成分の平行移動値.
    //! @param [in]     tz          Z成分の平行移動値.
    //! @return     平行移動行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateTranslation( const f32 tx, const f32 ty, const f32 tz );

    //--------------------------------------------------------------------------
    //! @brief      平行移動行列を生成します.
    //!
    //! @param [in]     tx          X成分の平行移動値.
    //! @param [in]     ty          Y成分の平行移動値.
    //! @param [in]     tz          Z成分の平行移動値.
    //! @param [out]    result      平行移動行列.
    //--------------------------------------------------------------------------
    static void    CreateTranslation( const f32 tx, const f32 ty, const f32 tz, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      平行移動行列を生成します.
    //!
    //! @param [in]     translate   平行移動値.
    //! @return     平行移動行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateTranslation( const Vector3& translate );

    //--------------------------------------------------------------------------
    //! @brief      平行移動行列を生成します.
    //!
    //! @param [in]     trnaslate   平行移動値.
    //! @param [out]    result      平行移動行列.
    //--------------------------------------------------------------------------
    static void    CreateTranslation( const Vector3& translate, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      X軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン).
    //! @return     X軸回りの回転行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateRotationX( const f32 radian );

    //--------------------------------------------------------------------------
    //! @brief      X軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン).
    //! @param [out]    result      X軸回りの回転行列.
    //--------------------------------------------------------------------------
    static void    CreateRotationX( const f32 radian, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      Y軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン).
    //! @return     Y軸回りの回転行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateRotationY( const f32 radian );

    //--------------------------------------------------------------------------
    //! @brief      Y軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン).
    //! @param [out]    result      Y軸回りの回転行列.
    //--------------------------------------------------------------------------
    static void    CreateRotationY( const f32 radian, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      Z軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン).
    //! @return     Z軸回りの回転行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateRotationZ( const f32 radian );

    //--------------------------------------------------------------------------
    //! @brief      Z軸回りの回転行列を生成します.
    //!
    //! @param [in]     radian         角度(ラジアン),.
    //! @param [out]    result      Z軸周りの回転行列.
    //--------------------------------------------------------------------------
    static void    CreateRotationZ( const f32 radian, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      四元数から行列を生成します.
    //!
    //! @param [in]     value       四元数.
    //! @return     四元数から生成された行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateFromQuaternion( const Quaternion& qua );

    //--------------------------------------------------------------------------
    //! @brief      四元数から行列を生成します.
    //!
    //! @param [in]     value       四元数.
    //! @param [out]    result      四元数から生成された行列.
    //--------------------------------------------------------------------------
    static void    CreateFromQuaternion( const Quaternion& qua, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された軸と角度から回転行列を生成します.
    //!
    //! @param [in]     axis        回転軸.
    //! @param [in]     radian      回転角(ラジアン).
    //! @return     回転行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateFromAxisAngle( const Vector3& axis, const f32 radian );

    //--------------------------------------------------------------------------
    //! @brief      指定された軸と角度から回転行列を生成します.
    //!
    //! @param [in]     axis        回転軸.
    //! @param [in]     radian      回転角(ラジアン).
    //! @param [out]    result      回転行列.
    //--------------------------------------------------------------------------
    static void    CreateFromAxisAngle( const Vector3& axis, const f32 radian, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      ヨー・ピッチ・ロール角から回転行列を生成します.
    //!
    //! @param [in]     yaw         ヨー角(ラジアン).
    //! @param [in]     pitch       ピッチ角(ラジアン).
    //! @param [in]     roll        ロール角(ラジアン).
    //! @return     回転行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateRotationFromYawPitchRoll( const f32 yaw, const f32 pitch, const f32 roll );

    //--------------------------------------------------------------------------
    //! @brief      ヨー・ピッチ・ロール角から回転行列を生成します.
    //!
    //! @param [in]     yaw         ヨー角(ラジアン).
    //! @param [in]     pitch       ピッチ角(ラジアン).
    //! @param [in]     roll        ロール角(ラジアン).
    //! @param [out]    result      回転行列.
    //--------------------------------------------------------------------------
    static void    CreateRotationFromYawPitchRoll( const f32 yaw, const f32 pitch, const f32 roll, Matrix& result );

    //--------------------------------------------------------------------------
    //! @brief      ビュー行列を生成します.
    //!
    //! @param [in]     position    カメラ位置.
    //! @param [in]     target      注視点.
    //! @param [in]     upward      上向きベクトル.
    //! @return     ビュー行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateLookAt( const Vector3& position, const Vector3& target, const Vector3& upward );

    //--------------------------------------------------------------------------
    //! @brief      ビュー行列を生成します.
    //!
    //! @param [in]     position    カメラ位置.
    //! @param [in]     target      注視点.
    //! @param [in]     upward      上向きベクトル.
    //! @param [out]    result      ビュー行列.
    //--------------------------------------------------------------------------
    static void    CreateLookAt( const Vector3& position, const Vector3& target, const Vector3& upward, Matrix &result );

    static Matrix  CreateLookTo( const Vector3& position, const Vector3& direction, const Vector3& upward );

    static void    CreateLookTo( const Vector3& position, const Vector3& direction, const Vector3& upward, Matrix& result );

    //--------------------------------------------------------------------------
    //! @brief      透視投影行列を生成します.
    //!
    //! @param [in]     width       ビューボリュームの幅.
    //! @param [in]     height      ビューボリュームの高さ.
    //! @param [in]     naerClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @return     透視投影行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreatePerspective( const f32 width, const f32 height, const f32 nearClip, const f32 farClip );

    //--------------------------------------------------------------------------
    //! @brief      透視投影行列を生成します.
    //!
    //! @param [in]     width       ビューボリュームの幅.
    //! @param [in]     height      ビューボリュームの高さ.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @param [out]    result      透視投影行列.
    //--------------------------------------------------------------------------
    static void    CreatePerspective( const f32 width, const f32 height, const f32 nearClip, const f32 farClip, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      視野角に基づいて透視投影行列を生成します.
    //!
    //! @param [in]     fieldOfView     視野角(ラジアン).
    //! @param [in]     aspectRatio     アスペクト比.
    //! @param [in]     nearClip        近クリップ平面までの距離.
    //! @param [in]     farClip         遠クリップ平面までの距離.
    //! @return     透視投影行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreatePerspectiveFieldOfView( const f32 fieldOfView, const f32 aspectRatio, const f32 nearClip, const f32 farClip );

    //--------------------------------------------------------------------------
    //! @brief      視野角に基づいて透視投影行列を生成します.
    //!
    //! @param [in]     fieldOfView     視野角(ラジアン).
    //! @param [in]     aspectRatio     アスペクト比.
    //! @param [in]     nearClip        近クリップ平面までの距離.
    //! @param [in]     farClip         遠クリップ平面までの距離.
    //! @param [out]    result          透視投影行列.
    //--------------------------------------------------------------------------
    static void    CreatePerspectiveFieldOfView( const f32 fieldOfView, const f32 aspectRatio, const f32 nearClip, const f32 farClip, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      カスタマイズした透視投影行列を生成します.
    //!
    //! @param [in]     left        ビューボリュームの最小X値.
    //! @param [in]     right       ビューボリュームの最大X値.
    //! @param [in]     bottom      ビューボリュームの最小Y値.
    //! @param [in]     top         ビューボリュームの最大Y値.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @return     透視投影行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreatePerspectiveOffcenter( const f32 left, const f32 right, const f32 bottom, const f32 top, const f32 nearClip, const f32 farClip );

    //--------------------------------------------------------------------------
    //! @brief      カスタマイズした透視投影行列を生成します.
    //!
    //! @param [in]     left        ビューボリュームの最小X値.
    //! @param [in]     right       ビューボリュームの最大X値.
    //! @param [in]     bottom      ビューボリュームの最小Y値.
    //! @param [in]     top         ビューボリュームの最大Y値.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @param [out]    result      透視投影行列.
    //--------------------------------------------------------------------------
    static void    CreatePerspectiveOffcenter( const f32 left, const f32 right, const f32 bottom, const f32 top, const f32 nearClip, const  f32 farClip, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      正射影行列を生成します.
    //!
    //! @param [in]     width       ビューボリュームの幅.
    //! @param [in]     height      ビューボリュームの高さ.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @return     正射影行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateOrthographic( const f32 width, const f32 height, const f32 nearClip, const f32 farClip );

    //--------------------------------------------------------------------------
    //! @brief      正射影行列を生成します.
    //!
    //! @param [in]     width       ビューボリュームの幅.
    //! @param [in]     height      ビューボリュームの高さ.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @param [out]    result      正射影行列.
    //--------------------------------------------------------------------------
    static void    CreateOrthographic( const f32 width, const f32 height, const f32 nearClip, const f32 farClip, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      カスタマイズした正射影行列を生成します.
    //!
    //! @param [in]     left        ビューボリュームの最小X値.
    //! @param [in]     right       ビューボリュームの最大X値.
    //! @param [in]     bottom      ビューボリュームの最小Y値.
    //! @param [in]     top         ビューボリュームの最大Y値.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @return     正射影行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  CreateOrthographicOffcenter( const f32 left, const f32 right, const f32 bottom, const f32 top, const f32 nearClip, const f32 farClip );

    //--------------------------------------------------------------------------
    //! @brief      カスタマイズした正射影行列を生成します.
    //!
    //! @param [in]     left        ビューボリュームの最小X値.
    //! @param [in]     right       ビューボリュームの最大X値.
    //! @param [in]     bottom      ビューボリュームの最小Y値.
    //! @param [in]     top         ビューボリュームの最大Y値.
    //! @param [in]     nearClip    近クリップ平面までの距離.
    //! @param [in]     farClip     遠クリップ平面までの距離.
    //! @param [out]    result      正射影行列.
    //--------------------------------------------------------------------------
    static void    CreateOrthographicOffcenter( const f32 left, const f32 right, const f32 bottom, const f32 top, const f32 nearClip, const f32 farClip, Matrix &result );

    //--------------------------------------------------------------------------
    //! @brief      2つの行列を線形補間します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @param [in]     amount      補間係数.
    //! @return     線形補間した行列を返却します.
    //--------------------------------------------------------------------------
    static Matrix  Lerp( const Matrix& a, const Matrix& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      2つの行列を線形補間します.
    //!
    //! @param [in]     a           入力行列.
    //! @param [in]     b           入力行列.
    //! @param [in]     amount      補間係数.
    //! @param [out]    result      線形補間された行列.
    //--------------------------------------------------------------------------
    static void    Lerp( const Matrix& a, const Matrix& b, const f32 amount, Matrix &result );

} Matrix;


////////////////////////////////////////////////////////////////////////////////
// Quaternion structure
////////////////////////////////////////////////////////////////////////////////
typedef struct Quaternion
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief          乗算演算子です.
    //!
    //! @param [in]     scalar          乗算するスカラー値.
    //! @param [in]     value           乗算される四元数.
    //! @return         乗算結果を返却します.
    //--------------------------------------------------------------------------
    friend Quaternion operator * ( f32, const Quaternion& );

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    f32 x;  //!< X成分です.
    f32 y;  //!< Y成分です.
    f32 z;  //!< Z成分です.
    f32 w;  //!< W成分です.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Quaternion();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValues     要素数4の配列.
    //--------------------------------------------------------------------------
    Quaternion( const f32* );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nx          X成分.
    //! @param [in]     ny          Y成分.
    //! @param [in]     nz          Z成分.
    //! @param [in]     nw          W成分.
    //--------------------------------------------------------------------------
    Quaternion( const f32 nx, const f32 ny, const f32 nz, const f32 nw );

    //--------------------------------------------------------------------------
    //! @brief      f32*型へのキャストです.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator       f32* ();

    //--------------------------------------------------------------------------
    //! @brief      const f32*型へのキャストです.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      加算代入演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& operator += ( const Quaternion& );

    //--------------------------------------------------------------------------
    //! @brief      減算代入演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& operator -= ( const Quaternion& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     value       乗算する値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& operator *= ( const Quaternion& );

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      スカラー乗算する値.
    //! @return     スカラー乗算した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& operator *= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      スカラー除算する値.
    //! @return     スカラー除算した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& operator /= ( f32 );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身の値を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator + () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //!
    //! @return     各成分の符号を反転した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator - () const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     value       乗算する値.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator *  ( const Quaternion& ) const;

    //--------------------------------------------------------------------------
    //! @brief      加算演算子です.
    //!
    //! @param [in]     value       加算する値.
    //! @return     加算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator +  ( const Quaternion& ) const;

    //--------------------------------------------------------------------------
    //! @brief      減算演算子です.
    //!
    //! @param [in]     value       減算する値.
    //! @return     減算結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator -  ( const Quaternion& ) const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      スカラー乗算する値.
    //! @return     スカラー乗算した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator *  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      スカラー除算する値.
    //! @return     スカラー除算した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion  operator /  ( f32 ) const;

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool        operator == ( const Quaternion& ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval flase   等価です.
    //--------------------------------------------------------------------------
    bool        operator != ( const Quaternion& ) const;

    //--------------------------------------------------------------------------
    //! @brief      四元数の長さを求めます.
    //!
    //! @return     四元数の長さを返却します.
    //--------------------------------------------------------------------------
    f32         Length       () const;

    //--------------------------------------------------------------------------
    //! @brief      四元数の長さの2乗値を求めます.
    //!
    //! @return     四元数の長さの2乗値を返却します.
    //--------------------------------------------------------------------------
    f32         LengthSq     () const;

    //--------------------------------------------------------------------------
    //! @brief      四元数の共役を求めます.
    //!
    //! @return     四元数の共役化した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& Conjugate    ();

    //--------------------------------------------------------------------------
    //! @brief     四元数を連結します.
    //!
    //! @param [in]     value       連結する四元数.
    //! @return     最初の回転とvalueによる回転を表す四元数を返却します.
    //--------------------------------------------------------------------------
    Quaternion& Concatenate  ( const Quaternion& );

    //--------------------------------------------------------------------------
    //! @brief      四元数を正規化します.
    //!
    //! @return     正規化した四元数を返却します.
    //--------------------------------------------------------------------------
    Quaternion& Normalize    ();

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して，四元数を正規化を試みます.
    //!
    //! @param [in]     set         長さが0の場合に設定する四元数.
    //! @return     長さが0でなければ正規化した四元数，0であればsetを返却します.
    //--------------------------------------------------------------------------
    Quaternion& SafeNormalize( const Quaternion& );

    //--------------------------------------------------------------------------
    //! @brief      単位四元数化します.
    //!
    //! @return     単位四元数化した結果を返却します.
    //--------------------------------------------------------------------------
    Quaternion& Identity     ();


    //--------------------------------------------------------------------------
    //! @brief      単位四元数にします.
    //!
    //! @param [in/out]     value   単位四元数化する値
    //--------------------------------------------------------------------------
    static void        Identity( Quaternion &value );

    //--------------------------------------------------------------------------
    //! @brief      単位四元数かどうかチェックします.
    //!
    //! @param [in]     value       チェックする値.
    //! @retval true    単位四元数です.
    //! @retval false   非単位四元数です.
    //--------------------------------------------------------------------------
    static bool        IsIdentity( const Quaternion &value );

    //--------------------------------------------------------------------------
    //! @brief      四元数同士の乗算を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @return     乗算結果を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Multiply( const Quaternion& a, const Quaternion& b );

    //--------------------------------------------------------------------------
    //! @brief      四元数同士の乗算を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [out]    result      乗算結果.
    //--------------------------------------------------------------------------
    static void        Multiply( const Quaternion& a, const Quaternion& b, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      四元数の内積を求めます.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @return     四元数の内積を返却します.
    //--------------------------------------------------------------------------
    static f32         Dot( const Quaternion& a, const Quaternion& b );

    //--------------------------------------------------------------------------
    //! @brief      四元数の内積を求めます.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [out]    result      四元数の内積.
    //--------------------------------------------------------------------------
    static void        Dot( const Quaternion& a, const Quaternion& b, f32 &result );

    //--------------------------------------------------------------------------
    //! @brief      四元数の共役を求めます.
    //!
    //! @param [in]     value       共役を求めたい四元数.
    //! @return     四元数の共役を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Conjugate( const Quaternion& value );

    //--------------------------------------------------------------------------
    //! @brief      四元数の共役を求めます.
    //!
    //! @param [in]     value       共役を求めたい四元数.
    //! @param [out]    result      四元数の共役.
    //--------------------------------------------------------------------------
    static void        Conjugate( const Quaternion& value, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      2つの四元数を連結します.
    //!
    //! @parma [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @return     aによる回転とbによる回転を表す四元数を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Concatenate( const Quaternion& a, const Quaternion& b );

    //--------------------------------------------------------------------------
    //! @brief      2つの四元数を連結します.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [out]    result      aによる回転とbによる回転を表す四元数.
    //--------------------------------------------------------------------------
    static void        Concatenate( const Quaternion& a, const Quaternion& b, Quaternion& result );

    //--------------------------------------------------------------------------
    //! @brief      四元数を正規化します.
    //!
    //! @param [in]     value       入力四元数.
    //! @return     四元数を正規化した結果を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Normalize( const Quaternion& value );

    //--------------------------------------------------------------------------
    //! @brief      四元数を正規化します.
    //!
    //! @param [in]     value       入力四元数.
    //! @param [out]    result      正規化した四元数.
    //--------------------------------------------------------------------------
    static void        Normalize( const Quaternion& value, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して正規化を試みます.
    //!
    //! @param [in]     value       入力四元数.
    //! @param [in]     set         長さが0の場合に設定する四元数.
    //! @return     長さが0でなければ正規化した四元数、長さが0であればsetを返却します.
    //--------------------------------------------------------------------------
    static Quaternion   SafeNormalize( const Quaternion& value, const Quaternion& set );

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して正規化を試みます.
    //!
    //! @param [in]     value       入力四元数.
    //! @param [in]     set         長さが0の場合に設定する四元数.
    //! @param [out]    result      長さが0でなければ正規化した四元数、長さが0であればset.
    //--------------------------------------------------------------------------
    static void         SafeNormalize( const Quaternion& value, const Quaternion& set, Quaternion& result );

    //--------------------------------------------------------------------------
    //! @brief      逆四元数を求めます.
    //!
    //! @param [in]     value       入力四元数.
    //! @return     逆四元数を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Inverse( const Quaternion& value );

    //--------------------------------------------------------------------------
    //! @brief      逆四元数を求めます.
    //!
    //! @param [in]     value       入力四元数.
    //! @param [out]    result      逆四元数.
    //--------------------------------------------------------------------------
    static void        Inverse( const Quaternion& value, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      ヨー・ピッチ・ロール角から四元数を生成します.
    //!
    //! @param [in]     yaw         ヨー角(ラジアン).
    //! @param [in]     pitch       ピッチ角(ラジアン).
    //! @param [in]     roll        ロール角(ラジアン).
    //! @return     指定されたヨー・ピッチ・ロール角から生成された四元数を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  CreateFromYawPitchRoll( const f32 yaw, const f32 pitch, const f32 roll );

    //--------------------------------------------------------------------------
    //! @brief      ヨー・ピッチ・ロール角から四元数を生成します.
    //!
    //! @param [in]     yaw         ヨー角(ラジアン).
    //! @param [in]     pitch       ピッチ角(ラジアン).
    //! @param [in]     roll        ロール角(ラジアン).
    //! @param [out]    result      指定されたヨー・ピッチ・ロール角から生成された四元数.
    //--------------------------------------------------------------------------
    static void        CreateFromYawPitchRoll( const f32 yaw, const f32 pitch, const f32 roll, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      指定された軸と角度から四元数を生成します.
    //!
    //! @param [in]     axis        回転軸.
    //! @param [in]     rad         回転角(ラジアン).
    //! @return     指定された軸と角度から生成された四元数を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  CreateFromAxisAngle( const Vector3& axis, const f32 radian );

    //--------------------------------------------------------------------------
    //! @brief      指定された軸と角度から四元数を生成します.
    //!
    //! @param [in]     axis        回転軸.
    //! @param [in]     rad         回転角(ラジアン).
    //! @param [out]    result      指定された軸と角度から生成された四元数.
    //--------------------------------------------------------------------------
    static void        CreateFromAxisAngle( const Vector3& axis, const f32 radian, Quaternion& result );

    //--------------------------------------------------------------------------
    //! @brief      回転行列から四元数を生成します.
    //!
    //! @param [in]     value       回転行列.
    //! @return     回転行列から生成した四元数を返却します.
    //--------------------------------------------------------------------------
    static Quaternion   CreateFromRotationMatrix( const Matrix& value );

    //--------------------------------------------------------------------------
    //! @brief      回転行列から四元数を生成します.
    //!
    //! @param [in]     value       回転行列.
    //! @param [out]    result      回転行列から生成した四元数を返却します.
    //--------------------------------------------------------------------------
    static void         CreateFromRotationMatrix( const Matrix& value, Quaternion& result );

    //--------------------------------------------------------------------------
    //! @brief      球面線形補間を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [in]     amount      補間係数.
    //! @return     球面線形補間した結果を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Slerp( const Quaternion& a, const Quaternion& b, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      球面線形補間を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [in]     amount      補間係数.
    //! @param [out]    result      球面線形補完した結果.
    //--------------------------------------------------------------------------
    static void        Slerp( const Quaternion& a, const Quaternion& b, const f32 amount, Quaternion &result );

    //--------------------------------------------------------------------------
    //! @brief      球面四角形補間を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [in]     c           入力四元数.
    //! @param [in]     d           入力四元数.
    //! @param [in]     amount      補間係数.
    //! @return     球面四角形補間した結果を返却します.
    //--------------------------------------------------------------------------
    static Quaternion  Squad( const Quaternion& value, const Quaternion& a, const Quaternion& b, const Quaternion& c, const f32 amount );

    //--------------------------------------------------------------------------
    //! @brief      球面四角形補間を行います.
    //!
    //! @param [in]     a           入力四元数.
    //! @param [in]     b           入力四元数.
    //! @param [in]     c           入力四元数.
    //! @param [in]     d           入力四元数.
    //! @param [in]     amount      補間係数.
    //! @param [out]    result      球面四角形補間した結果.
    //--------------------------------------------------------------------------
    static void        Squad( const Quaternion& value, const Quaternion& a, const Quaternion& b, const Quaternion& c, const f32 amount, Quaternion &result );

} Quaternion;



} // namespace asdx

//-----------------------------------------------------------------------
// Inline Files
//-----------------------------------------------------------------------
#include "asdxMath.inl"


#endif//__PHANTOM_MATH_H__