﻿//-------------------------------------------------------------------------------------
// File : asdxGeometry.h
// Desc : Geometry Module
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------

#ifndef __ASDX_GEOMETRY_H__
#define __ASDX_GEOMETRY_H__


//-------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------
#include <asdxMath.h>


namespace asdx {

//-------------------------------------------------------------------------------------
// Forward Declarations
//-------------------------------------------------------------------------------------
class Plane;
class Ray;
class BoundingBox;
class BoundingSphere;
class BoundingFrustum;


///////////////////////////////////////////////////////////////////////////////
// ConstainmentType enum
///////////////////////////////////////////////////////////////////////////////
enum ContainmentType
{
    DISJOINT    = 0,    //!< 境界ボリューム同士の重なりが無いこと示します.
    INTERSECTS  = 1,    //!< 境界ボリューム同士がぶ部分的に重なり合っていることを示します.
    CONTAINS    = 2,    //!< ある境界ボリュームが別の境界ボリュームに完全に包むことを示します.
};


////////////////////////////////////////////////////////////////////////////////
// PlaneIntersectionType enum
////////////////////////////////////////////////////////////////////////////////
enum PlaneIntersectionType
{
    FRONT           = 0,    //!< 交差はありません.境界空間は平面のプラス側(半空間)にあります.
    INTERSECTING    = 1,    //!< 平面は交差しています.
    BACK            = 2,    //!< 交差はありません.境界空間は平面のマイナス側(半空間)にあります.
};

////////////////////////////////////////////////////////////////////////////////
// Vector3x8 class
////////////////////////////////////////////////////////////////////////////////
class Vector3x8
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================
    /* NOTHING */

private:
    //==========================================================================
    // private variables
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // private methods
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables
    //==========================================================================
    static const u32 NUM_COUNT = 8; //!< 配列の要素数(=8)です.
    Vector3 points[ NUM_COUNT ];        //!< 8要素の3次元ベクトルの配列です.

    //==========================================================================
    // protected methods.
    //==========================================================================
    /* NOTHING */

public:
    //==========================================================================
    // public variables
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // public methods
    //==========================================================================
    
    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Vector3x8();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     v0      1番目の点.
    //! @param [in]     v1      2番目の点.
    //! @param [in]     v2      3番目の点.
    //! @param [in]     v3      4番目の点.
    //! @param [in]     v4      5番目の点.
    //! @param [in]     v5      6番目の点.
    //! @param [in]     v6      7番目の点.
    //! @param [in]     v7      8番目の点.
    //--------------------------------------------------------------------------
    Vector3x8( 
        const Vector3& v0,
        const Vector3& v1,
        const Vector3& v2,
        const Vector3& v3,
        const Vector3& v4,
        const Vector3& v5,
        const Vector3& v6,
        const Vector3& v7 
    );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValues     要素数8の3次元ベクトル配列です.
    //--------------------------------------------------------------------------
    Vector3x8( Vector3* pValues );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    Vector3x8( const Vector3x8& value );

    //--------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //--------------------------------------------------------------------------
    virtual ~Vector3x8();

    //--------------------------------------------------------------------------
    //! @brief      インデクサです.
    //! 
    //! @param [in]     index       取得する要素番号.
    //! @return     指定された要素番号の3次元ベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector3&        operator [] ( u32 index );

    //--------------------------------------------------------------------------
    //! @brief      インデクサです(const版).
    //! 
    //! @param [in]     index       取得する要素番号.
    //! @return     指定された要素番号の3次元ベクトルを返却します.
    //--------------------------------------------------------------------------
    const Vector3&  operator [] ( u32 index ) const;

    //--------------------------------------------------------------------------
    //! @brief      サイズを取得します.
    //!
    //! @retval 8   常に8を返却します.
    //--------------------------------------------------------------------------
    u32         GetSize () const;

    //--------------------------------------------------------------------------
    //! @brief      指定された要素番号の3次元ベクトルの値を取得します.
    //!
    //! @param [in]     index       取得する要素番号.
    //! @return     指定された要素番号の3次元ベクトルを返却します.
    //--------------------------------------------------------------------------
    Vector3&    GetAt   ( u32 index );

    //--------------------------------------------------------------------------
    //! @brief      指定された要素番号の3次元ベクトルに値を設定します.
    //!
    //! @param [in]     index       設定する要素番号.
    //! @param [in]     value       設定する3次元ベクトルの値.
    //--------------------------------------------------------------------------
    void        SetAt   ( u32 index, const Vector3& value );

    //--------------------------------------------------------------------------
    //! @breif      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Vector3x8& operator =  ( const Vector3x8& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval flase   非等価です.
    //--------------------------------------------------------------------------
    bool       operator == ( const Vector3x8& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //--------------------------------------------------------------------------
    bool       operator != ( const Vector3x8& value ) const;
};


////////////////////////////////////////////////////////////////////////////////
// Plane class
////////////////////////////////////////////////////////////////////////////////
ASDX_ALIGN( 16 )
class Plane
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @param [in]     value       乗算される平面式.
    //! @return     スカラー乗算した結果を返却します.
    //--------------------------------------------------------------------------
    friend Plane operator * ( const f32 scalar, const Plane& value );

private:
    //==========================================================================
    // private variables.
    //==========================================================================
    /* NOTHING */
    
    //==========================================================================
    // private methods.
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // protected methods.
    //==========================================================================
    /* NOTHING */

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    Vector3 normal;     //!< 法線ベクトルです.
    f32     d;          //!< 原点からの距離です.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Plane();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pValues         要素数4の配列.
    //--------------------------------------------------------------------------
    Plane( const f32* pValues );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nx          法線ベクトルのX成分です.
    //! @param [in]     ny          法線ベクトルのY成分です.
    //! @param [in]     nz          法線ベクトルのZ成分です.
    //! @param [in]     nd          原点からの距離です.
    //--------------------------------------------------------------------------
    Plane( const f32 nx, const f32 ny, const f32 nz, const f32 nd );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @parma [in]     norm        法線ベクトルです.
    //! @param [in]     dist        原点からの距離です.
    //--------------------------------------------------------------------------
    Plane( const Vector3& norm, const f32 dist );

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     value       4次元ベクトル.
    //--------------------------------------------------------------------------
    Plane( const Vector4& value );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    Plane( const Plane& value );

    //--------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //--------------------------------------------------------------------------
    virtual ~Plane();

    //--------------------------------------------------------------------------
    //! @brief      f32*型へのキャストです.
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator        f32* ();

    //--------------------------------------------------------------------------
    //! @brief      f32*型へのキャストです(const版).
    //!
    //! @return     最初の要素へのポインタを返却します.
    //--------------------------------------------------------------------------
    operator const  f32* () const;

    //--------------------------------------------------------------------------
    //! @brief      乗算代入演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     スカラー乗算した結果を返却します.
    //--------------------------------------------------------------------------
    Plane&  operator *= ( const f32 scalar );

    //--------------------------------------------------------------------------
    //! @brief      除算代入演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     スカラー除算した結果を返却します.
    //--------------------------------------------------------------------------
    Plane&  operator /= ( const f32 scalar );

    //--------------------------------------------------------------------------
    //! @brief      正符号演算子です.
    //!
    //! @return     自分自身の値を返却します.
    //--------------------------------------------------------------------------
    Plane   operator +  () const;

    //--------------------------------------------------------------------------
    //! @brief      負符号演算子です.
    //!
    //! @return     負符号を付けた値を返却します.
    //--------------------------------------------------------------------------
    Plane   operator -  () const;

    //--------------------------------------------------------------------------
    //! @brief      乗算演算子です.
    //!
    //! @param [in]     scalar      乗算するスカラー値.
    //! @return     スカラー乗算した結果を返却します.
    //--------------------------------------------------------------------------
    Plane   operator *  ( const f32 scalar ) const;

    //--------------------------------------------------------------------------
    //! @brief      除算演算子です.
    //!
    //! @param [in]     scalar      除算するスカラー値.
    //! @return     スカラー除算した結果を返却します.
    //--------------------------------------------------------------------------
    Plane   operator /  ( const f32 scalar ) const;

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Plane&  operator =  ( const Plane& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool    operator == ( const Plane& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool    operator != ( const Plane& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      正規化を行います.
    //!
    //! @return     正規化した結果を返却します.
    //--------------------------------------------------------------------------
    Plane&  Normalize       ();

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して正規化を試みます.
    //!
    //! @param [in]     set         長さが0の場合に設定する値.
    //! @return     長さが0でなければ正規化した結果,そうでなければsetを返却します.
    //--------------------------------------------------------------------------
    Plane&  SafeNormalize   ( const Plane& set );

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       内積を求める4次元ベクトル.
    //! @return     4次元ベクトルとの内積の値を返却します.
    //--------------------------------------------------------------------------
    f32     Dot             ( const Vector4& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      内積の値に平面の距離を足した値を求めます.
    //!
    //! @param [in]     value       内積を求める3次元ベクトル.
    //! @return     3次元ベクトルとの内積に平面の距離を足した値を返却します.
    //--------------------------------------------------------------------------
    f32     DotCoordinate   ( const Vector3& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       内積を求める3次元ベクトル.
    //! @return     3次元ベクトルとの内積の値を返却します.
    //--------------------------------------------------------------------------
    f32     DotNormal       ( const Vector3& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       点
    //! @return     点との交差判定結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const Vector3& value )    const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界箱
    //! @return     境界箱との交差判定結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const BoundingBox& value )      const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界錐台
    //! @return     境界錐台との交差判定結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const BoundingFrustum& value )  const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界球
    //! @return     境界球との交差判定結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const BoundingSphere& value )   const;

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         4次元ベクトル
    //! @return     平面式と4次元ベクトルとの内積の値を返却します.
    //--------------------------------------------------------------------------
    static f32      Dot( const Plane& value, const Vector4& vec );

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         4次元ベクトル.
    //! @param [out]    result      内積.
    //--------------------------------------------------------------------------
    static void     Dot( const Plane& value, const Vector4& vec, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      内積の値に平面の距離を足した値を求めます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         3次元ベクトル.
    //! @return     平面式と3次元ベクトルとの内積の値に平面の距離を足した値を返却します.
    //--------------------------------------------------------------------------
    static f32      DotCoordinate( const Plane& value, const Vector3& vec );

    //--------------------------------------------------------------------------
    //! @brief      内積の値に平面の距離を足した値を返却します.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         3次元ベクトル.
    //! @param [out]    result      内積の値に平面の距離を足した値.
    //--------------------------------------------------------------------------
    static void     DotCoordinate( const Plane& value, const Vector3& vec, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         3次元ベクトル.
    //! @return     平面の法線ベクトルと3次元ベクトルとの内積の値を返却します.
    //--------------------------------------------------------------------------
    static f32      DotNormal( const Plane& value, const Vector3& vec );

    //--------------------------------------------------------------------------
    //! @brief      内積の値を求めます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     vec         3次元ベクトル.
    //! @param [out]    resul       内積.
    //--------------------------------------------------------------------------
    static void     DotNormal( const Plane& value, const Vector3& vec, f32& result );

    //--------------------------------------------------------------------------
    //! @brief      正規化を行います.
    //!
    //! @param [in]     value       平面式.
    //! @return     正規化した結果を返却します.
    //--------------------------------------------------------------------------
    static Plane    Normalize( const Plane& value );

    //--------------------------------------------------------------------------
    //! @brief      正規化を行います.
    //!
    //! @param [in]     value       平面式.
    //! @param [out]    result      正規化した結果を返却します.
    //--------------------------------------------------------------------------
    static void     Normalize( const Plane& value, Plane& result );

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して，正規化を試みます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     set         長さが0の場合に設定する値.
    //! @return     長さが0でなければ正規化した結果を，0であればsetを返却します.
    //--------------------------------------------------------------------------
    static Plane    SafeNormalize( const Plane& value, const Plane& set );

    //--------------------------------------------------------------------------
    //! @brief      零除算を考慮して，正規化を試みます.
    //!
    //! @param [in]     value       平面式.
    //! @param [in]     set         長さが0の場合の設定する値.
    //! @param [out]    result      長さが0でなければ正規化した値，0であればset.
    //--------------------------------------------------------------------------
    static void     SafeNormalize( const Plane& value, const Plane& set, Plane& result );
};




////////////////////////////////////////////////////////////////////////////////
// Ray class
////////////////////////////////////////////////////////////////////////////////
class Ray
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================
    /* NOTHING */

private:
    //==========================================================================
    // private variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // private methods.
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // protected methods.
    //==========================================================================
    /* NOTHING */

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    Vector3 position;           //!< 位置.
    Vector3 direction;          //!< 方向.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    Ray();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     pos         位置ベクトル.
    //! @param [in]     dir         方向ベクトル.
    //--------------------------------------------------------------------------
    Ray( const Vector3& pos, const Vector3& dir );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    Ray( const Ray& value );

    //--------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //--------------------------------------------------------------------------
    virtual ~Ray();

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    Ray& operator =     ( const Ray& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool operator ==    ( const Ray& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //--------------------------------------------------------------------------
    bool operator !=    ( const Ray& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界箱.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool  Intersects( const BoundingBox& value, f32& distance )       const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界錐台
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool  Intersects( const BoundingFrustum& value, f32& distance )   const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       境界球.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool  Intersects( const BoundingSphere& value, f32& distance )    const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       平面式.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool  Intersects( const Plane& value, f32& distance )             const;

    //---------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     p0          三角形を構成する点0.
    //! @param [in]     p1          三角形を構成する点1.
    //! @param [in]     p2          三角形を構成する点2.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //---------------------------------------------------------------------------
    bool  Intersects( const Vector3& p0, const Vector3& p1, const Vector3& p2, f32& distance ) const;
};


////////////////////////////////////////////////////////////////////////////////
// BoundingBox class
////////////////////////////////////////////////////////////////////////////////
class BoundingBox
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================
    /* NOTHING */

private:
    //==========================================================================
    // private variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // private methods.
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // protected methods.
    //==========================================================================
    /* NOTHING */

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    Vector3     mini;        //!< 最小値です.
    Vector3     maxi;        //!< 最大値です.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    BoundingBox();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     mini        最小値.
    //! @param [in]     maxi        最大値.
    //--------------------------------------------------------------------------
    BoundingBox( const Vector3& mini, const Vector3& maxi );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    BoundingBox( const BoundingBox& value );

    //--------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //--------------------------------------------------------------------------
    virtual ~BoundingBox();

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    BoundingBox&    operator =  ( const BoundingBox& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool            operator == ( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //--------------------------------------------------------------------------
    bool            operator != ( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する点.
    //! @return     境界箱に点が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const Vector3& value )          const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界箱.
    //! @return     境界箱に境界箱が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingBox& value )      const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @return     境界箱に境界球が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingSphere& value )   const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界錐台.
    //! @return     境界箱に境界錐台が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingFrustum& value )  const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界箱.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界錐台.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する平面式.
    //! @retval     交差判定の結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const Plane& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定するレイ.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const Ray& value, f32& distance ) const;

    //--------------------------------------------------------------------------
    //! @brief      境界箱を構成する8角の点を取得します.
    //!
    //! @return     境界箱を構成する8角の点を返却します.
    //--------------------------------------------------------------------------
    Vector3x8 GetCorners() const;

    //--------------------------------------------------------------------------
    //! @brief      境界箱を構成する8角の点を取得します.
    //!
    //! @param [out]    result      境界箱を構成する8角の点.
    //--------------------------------------------------------------------------
    void GetCorners( Vector3x8& result ) const;

    //--------------------------------------------------------------------------
    //! @brief      2つの境界箱をマージします.
    //!
    //! @param [in]     a           入力境界箱.
    //! @param [in]     b           入力境界箱.
    //! @return     2つの境界箱をマージした結果を返却します.
    //--------------------------------------------------------------------------
    static BoundingBox CreateMerged( const BoundingBox& a, const BoundingBox& b );

    //--------------------------------------------------------------------------
    //! @brief      2つの境界箱をマージします.
    //!
    //! @param [in]     a           入力境界箱.
    //! @param [in]     b           入力境界箱.
    //! @param [out]    result      マージした境界箱.
    //--------------------------------------------------------------------------
    static void CreateMerged( const BoundingBox& a, const BoundingBox& b, BoundingBox& result );

    //--------------------------------------------------------------------------
    //! @brief      境界球から境界箱を生成します.
    //!
    //! @param [in]     value       境界球.
    //! @return     境界球から作成した境界箱を返却します.
    //--------------------------------------------------------------------------
    static BoundingBox CreateFromSphere( const BoundingSphere& value );

    //--------------------------------------------------------------------------
    //! @brief      境界球から境界箱を生成します.
    //!
    //! @param [in]     value       境界球.
    //! @param [out]    result      作成した境界箱.
    //--------------------------------------------------------------------------
    static void CreateFromSphere( const BoundingSphere& value, BoundingBox& result );

    //--------------------------------------------------------------------------
    //! @brief      点群から境界箱を生成します.
    //!
    //! @param [in]     numPoints       点の数.
    //! @param [in]     pPoints         点群データの配列.
    //! @param [in]     offset          配列の先頭からのオフセット.
    //! @return     点群から生成した境界箱を返却します.
    //--------------------------------------------------------------------------
    static BoundingBox CreateFromPoints( const u32 numPoints, const Vector3* pPoints, const u32 offset );

    //--------------------------------------------------------------------------
    //! @brief      点群から境界箱を生成します.
    //!
    //! @param [in]     numPoints       点の数.
    //! @param [in]     pPoints         点群データの配列.
    //! @param [in]     offset          配列の先頭からのオフセット.
    //! @param [out]    result          生成した境界箱.
    //--------------------------------------------------------------------------
    static void CreateFromPoints( const u32 numPoints, const Vector3* pPoints, const u32 offset, BoundingBox& result );

    //--------------------------------------------------------------------------
    //! @brief      8角の点から境界箱を生成します.
    //!
    //! @param [in]     coners          8角の点.
    //! @return     8角の点から生成した境界箱を返却します.
    //--------------------------------------------------------------------------
    static BoundingBox CreateFromCorners( const Vector3x8& corners );

    //--------------------------------------------------------------------------
    //! @brief      8角の点から境界箱を生成します.
    //!
    //! @param [in]     coners          8角の点.
    //! @param [out]    result          生成した境界箱.
    //--------------------------------------------------------------------------
    static void CreateFromCorners( const Vector3x8& corners, BoundingBox& result );
};


////////////////////////////////////////////////////////////////////////////////
// BoundingSphere class
////////////////////////////////////////////////////////////////////////////////
class BoundingSphere
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================
    /* NOTHING */

private:
    //==========================================================================
    // private variables.
    //==========================================================================
    /* DO_NOTHING */

    //==========================================================================
    // private methods.
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // protected methods.
    //==========================================================================
    /* NOTHING */

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    Vector3     center;     //!< 中心位置です.
    f32         radius;     //!< 半径です.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    BoundingSphere();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     nc      中心位置.
    //! @param [in]     nr      半径.
    //--------------------------------------------------------------------------
    BoundingSphere( const Vector3& nc, const f32 nr );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    BoundingSphere( const BoundingSphere& value );

    //--------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //--------------------------------------------------------------------------
    virtual ~BoundingSphere();

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    BoundingSphere& operator =  ( const BoundingSphere& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool operator == ( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //--------------------------------------------------------------------------
    bool operator != ( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する点.
    //! @return     境界球に点が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const Vector3& value )          const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @return     境界球に境界箱が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingBox& value )      const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @return     境界球に境界球が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingSphere& value )   const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界錐台.
    //! @return     境界球に境界錐台が含まれているかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingFrustum& value )  const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界箱.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界錐台.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する平面式.
    //! @return     交差判定の結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const Plane& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定するレイ.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const Ray& value, f32& distance ) const;

    //--------------------------------------------------------------------------
    //! @brief      2つの境界球をマージします.
    //!
    //! @param [in]     a           入力境界球.
    //! @param [in]     b           入力境界球.
    //! @return     2つの境界球をマージした結果を返却します.
    //--------------------------------------------------------------------------
    static BoundingSphere CreateMerged( const BoundingSphere& a, const BoundingSphere& b );

    //--------------------------------------------------------------------------
    //! @brief      2つの境界球をマージします.
    //!
    //! @param [in]     a           入力境界球.
    //! @param [in]     b           入力境界球.
    //! @param [out]    result      マージした境界球.
    //--------------------------------------------------------------------------
    static void CreateMerged( const BoundingSphere& a, const BoundingSphere& b, BoundingSphere& result );

    //--------------------------------------------------------------------------
    //! @brief      境界箱から境界球を生成します.
    //!
    //! @param [in]     value       境界箱.
    //! @return     境界箱から生成した境界球を返却します.
    //--------------------------------------------------------------------------
    static BoundingSphere CreateFromBoundingBox( const BoundingBox& value );

    //--------------------------------------------------------------------------
    //! @brief      境界箱から境界球を生成します.
    //!
    //! @param [in]     value       境界箱.
    //! @param [out]    result      生成した境界球.
    //--------------------------------------------------------------------------
    static void CreateFromBoundingBox( const BoundingBox& value, BoundingSphere& result );

    //--------------------------------------------------------------------------
    //! @brief      点群から境界球を生成します.
    //!
    //! @param [in]     numPoint    点の数.
    //! @param [in]     pPoints     点群データの配列.
    //! @param [in]     offset      配列の先頭からのオフセット.
    //! @return     点群から生成された境界球を返却します.
    //--------------------------------------------------------------------------
    static BoundingSphere CreateFromPoints( const u32 numPoint, const Vector3* pPoints, const u32 offset );

    //--------------------------------------------------------------------------
    //! @brief      点群から境界球を生成します.
    //!
    //! @param [in]     numPoint    点の数.
    //! @param [in]     pPoints     点群データの配列.
    //! @param [in]     offset      配列の先頭からのオフセット.
    //! @param [out]    result      生成された境界球.
    //--------------------------------------------------------------------------
    static void CreateFromPoints( const u32 numPoint, const Vector3* pPoints, const u32 offset, BoundingSphere& result );

    //--------------------------------------------------------------------------
    //! @brief      8角の点から境界球を生成します.
    //!
    //! @param [in]     corners     8角の点.
    //! @return     8角の点から生成された境界球を返却します.
    //--------------------------------------------------------------------------
    static BoundingSphere CreateFromCorners( const Vector3x8& coners );

    //--------------------------------------------------------------------------
    //! @brief      8角の点から境界球を生成します.
    //!
    //! @param [in]     corners     8角の点.
    //! @param [out:    sphere      生成された境界球.
    //--------------------------------------------------------------------------
    static void CreateFromCorners( const Vector3x8& coners, BoundingSphere& sphere );
};


////////////////////////////////////////////////////////////////////////////////
// BoundingFrustum class
////////////////////////////////////////////////////////////////////////////////
class BoundingFrustum
{
    //==========================================================================
    // list of friend classes and methods.
    //==========================================================================
    /* NOTHING */

private:
    //==========================================================================
    // private variables.
    //==========================================================================
    /* NOTHING */

    //==========================================================================
    // private methods.
    //==========================================================================
    /* NOTHING */

protected:
    //==========================================================================
    // protected variables.
    //==========================================================================
    Matrix      matrix;     //!< 行列です.

    //==========================================================================
    // protected methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      行列から6平面を算出します.
    //!
    //! @param [in]     mat             行列.
    //! @param [in]     leftPlane       左側面.
    //! @param [in]     rightPlane      右側面.
    //! @param [in]     topPlane        上面.
    //! @param [in]     bottomPlane     底面.
    //! @param [in]     nearPlane       近平面.
    //! @param [in]     farPlane        遠平面.
    //--------------------------------------------------------------------------
    static void     ComputePlanesFromMatrix( 
        const Matrix& mat,
        Plane& leftPlane,
        Plane& rightPlane,
        Plane& topPlane,
        Plane& bottomPlane,
        Plane& nearPlane,
        Plane& farPlane
    );

    //--------------------------------------------------------------------------
    //! @brief      3平面の交差点を求めます.
    //!
    //! @param [in]     a       入力平面.
    //! @param [in]     b       入力平面.
    //! @param [in]     c       入力平面.
    //! @return     3平面の交差点を返却します.
    //--------------------------------------------------------------------------
    static Vector3  IntersectionPoint( 
        const Plane& a,
        const Plane& b,
        const Plane& c
    );

public:
    //==========================================================================
    // public variables.
    //==========================================================================
    enum
    {
        PLANE_LEFT = 0,
        PLANE_RIGHT,
        PLANE_TOP,
        PLANE_BOTTOM,
        PLANE_NEAR,
        PLANE_FAR
    };
    Plane plane[ 6 ];           //!< 要素数6の平面式の配列です.

    //==========================================================================
    // public methods.
    //==========================================================================

    //--------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //--------------------------------------------------------------------------
    BoundingFrustum();

    //--------------------------------------------------------------------------
    //! @brief      引数付きコンストラクタです.
    //!
    //! @param [in]     mat     行列.
    //--------------------------------------------------------------------------
    BoundingFrustum( const Matrix& mat );

    //--------------------------------------------------------------------------
    //! @brief      コピーコンストラクタです.
    //!
    //! @param [in]     value       コピー元の値.
    //--------------------------------------------------------------------------
    BoundingFrustum( const BoundingFrustum& value );

    //--------------------------------------------------------------------------
    //! @brief      代入演算子です.
    //!
    //! @param [in]     value       代入する値.
    //! @return     代入結果を返却します.
    //--------------------------------------------------------------------------
    BoundingFrustum& operator =  ( const BoundingFrustum& value );

    //--------------------------------------------------------------------------
    //! @brief      等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    等価です.
    //! @retval false   非等価です.
    //--------------------------------------------------------------------------
    bool operator == ( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      非等価比較演算子です.
    //!
    //! @param [in]     value       比較する値.
    //! @retval true    非等価です.
    //! @retval false   等価です.
    //--------------------------------------------------------------------------
    bool operator != ( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する点
    //! @return     境界錘台に点が含まれるかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const Vector3& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @return     境界錐台に境界箱が含まれるかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界箱.
    //! @retval     境界箱に境界球が含まれるかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      包含判定を行います.
    //!
    //! @param [in]     value       判定する境界錘台.
    //! @retval     境界錘台に境界球が含まれるかどうかの判定結果を返却します.
    //--------------------------------------------------------------------------
    ContainmentType Contains( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界箱.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingBox& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界球.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingSphere& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する境界錘台.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const BoundingFrustum& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定する平面式.
    //! @return     交差判定の結果を返却します.
    //--------------------------------------------------------------------------
    PlaneIntersectionType Intersects( const Plane& value ) const;

    //--------------------------------------------------------------------------
    //! @brief      交差判定を行います.
    //!
    //! @param [in]     value       判定するレイ.
    //! @param [out]    distance    交差点までの距離.
    //! @retval true    交差しています.
    //! @retval false   交差はありません.
    //--------------------------------------------------------------------------
    bool Intersects( const Ray&, f32& distance ) const;

    //--------------------------------------------------------------------------
    //! @brief      境界錘台を構成する8角の点を取得します.
    //!
    //! @return     境界錘台を構成する8角の点を返却します.
    //--------------------------------------------------------------------------
    Vector3x8 GetCorners() const;

    //--------------------------------------------------------------------------
    //! @brief      境界錘台を構成する8角の点を取得します.
    //!
    //! @param [out]    result      境界錘台を構成する8角の点.
    //--------------------------------------------------------------------------
    void GetCorners( Vector3x8& result ) const;
};


//------------------------------------------------------------------------------
//! @brief      2次元上の三角形の面積を求めます.
//!
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @return     2次元上の三角形の面積を返却します.
//------------------------------------------------------------------------------
f32     ComputeTriangleArea( const Vector2& p0, const Vector2& p1, const Vector2& p2 );

//------------------------------------------------------------------------------
//! @brief      2次元上の三角形の面積を求めます.
//!
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @param [in]     result  2次元上の三角形の面積.
//------------------------------------------------------------------------------
void    ComputeTriangleArea( const Vector2& p0, const Vector2& p1, const Vector2& p2, f32& result );

//------------------------------------------------------------------------------
//! @brief      3次元上の三角形の面積を求めます.
//!
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @return     3次元上の三角形の面積を返却します.
//------------------------------------------------------------------------------
f32     ComputeTriangleArea( const Vector3& p0, const Vector3& p1, const Vector3& p2 );

//------------------------------------------------------------------------------
//! @brief      3次元上の三角形の面積を求めます.
//!
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @param [in]     result  3次元上の三角形の面積.
//------------------------------------------------------------------------------
void    ComputeTriangleArea( const Vector3& p0, const Vector3& p1, const Vector3& p2, f32& result );

//------------------------------------------------------------------------------
//! @brief      外接円の中心を求めます.
//!
//! @param [in]     p0      入力点.
//! @param [in]     p1      入力点.
//! @param [in]     p2      入力点.
//! @return     外接円の中心座標を返却します.
//------------------------------------------------------------------------------
Vector2 ComputeCircumcircleCenter( const Vector2& p0, const Vector2& p1, const Vector2& p2 );

//------------------------------------------------------------------------------
//! @brief      外接円の中心を求めます.
//!
//! @param [in]     p0      入力点.
//! @param [in]     p1      入力点.
//! @param [in]     p2      入力点.
//! @param [out]    result  外接円の中心座標.
//------------------------------------------------------------------------------
void    ComputeCircumcircleCenter( const Vector2& p0, const Vector2& p1, const Vector2& p2, Vector2& result );

//------------------------------------------------------------------------------
//! @brief      ドローネ三角形の条件を満たすかどうか判定します.
//!
//! @param [in]     value   判定点.
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @retval true    ドローネ三角形です.
//! @retval false   非ドローネ三角形です.
//------------------------------------------------------------------------------
bool    IsDelaunayTriangle( const Vector2& value, const Vector2& p0, const Vector2& p1, const Vector2& p2 );

//------------------------------------------------------------------------------
//! @brief      ドローネ三角形の条件を満たすかどうか判定します.
//!
//! @param [in]     value   判定点.
//! @param [in]     p0      三角形を構成する点.
//! @param [in]     p1      三角形を構成する点.
//! @param [in]     p2      三角形を構成する点.
//! @param [out]    result  ドローネ三角形であればtrue, 非ドローネ三角形であればfalse.
//------------------------------------------------------------------------------
void    IsDelaunayTriangle( const Vector2& value, const Vector2& p0, const Vector2& p1, const Vector2& p2, bool &result );


} // namespace asdx


//--------------------------------------------------------------------------------
// Inline Files
//--------------------------------------------------------------------------------
#include "asdxGeometry.inl"


#endif//__ASDX_GEOMETRY_H__