//--------------------------------------------------------------------------------------------
// File : asdxMesh.cpp
// Desc : Mesh Module.
// Copyright(c) Project Asura. All right reserved.
//---------------------------------------------------------------------------------------------

//---------------------------------------------------------------------------------------------
// Includes
//---------------------------------------------------------------------------------------------
#include <asdxMesh.h>
#include <cassert>
#include <cstdio>


namespace asdx {

//--------------------------------------------------------------------------------------------
// Constant Values
//--------------------------------------------------------------------------------------------

// ͗vfł.
const D3D11_INPUT_ELEMENT_DESC Mesh::INPUT_ELEMENTS[ Mesh::NUM_INPUT_ELEMENT ] =
{
    { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0 },
    { "NORMAL",   0, DXGI_FORMAT_R32G32B32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0 },
    { "TANGENT",  0, DXGI_FORMAT_R32G32B32_FLOAT, 0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0 }, 
    { "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT,    0, D3D11_APPEND_ALIGNED_ELEMENT, D3D11_INPUT_PER_VERTEX_DATA, 0 },
};

// ݂MSHt@C̃o[Wԍł.
static const unsigned int MSH_CURRENT_VERSION = 0x00000001;


/////////////////////////////////////////////////////////////////////////////////////////////
// MSH_DATA_HEADER structure
//! @brief  MSHt@C̃f[^wb_[ł.
/////////////////////////////////////////////////////////////////////////////////////////////
struct MSH_DATA_HEADER
{
    unsigned int NumVertices;               //!< _ł.
    unsigned int NumIndices;                //!< CfbNXł.
    unsigned int NumMaterials;              //!< }eAł.
    unsigned int NumSubsets;                //!< TuZbgł.
    unsigned int VertexStructureSize;       //!< _f[^\̂̃TCYł.
    unsigned int IndexStructureSize;        //!< _CfbNXf[^̍\̂ł.
    unsigned int MaterialStructureSize;     //!< }eAf[^\̂̃TCYł.
    unsigned int SubsetStructureSize;       //!< TuZbgf[^\̂̃TCYł.
};


/////////////////////////////////////////////////////////////////////////////////////////////
// MSH_FILE_HEADER structure
//! @brief  MSHt@C̃t@Cwb_[ł.
/////////////////////////////////////////////////////////////////////////////////////////////
struct MSH_FILE_HEADER
{
    unsigned char   Magic[ 4 ];         //!< }WbNł( 'M, 'S', 'H', '\0' )
    unsigned int    Version;            //!< t@Co[Wł.
    unsigned int    DataHeaderSize;     //!< f[^wb_[\̂̃TCYł.
    MSH_DATA_HEADER DataHeader;         //!< f[^wb_[ł.
};


/////////////////////////////////////////////////////////////////////////////////////////////
// Mesh class 
/////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------
//      RXgN^ł.
//-------------------------------------------------------------------------------------------
Mesh::Mesh()
: m_VertexCount     ( 0 )
, m_IndexCount      ( 0 )
, m_MaterialCount   ( 0 )
, m_SubsetCount     ( 0 )
, m_pVertex         ( nullptr )
, m_pIndex          ( nullptr )
, m_pMaterial       ( nullptr )
, m_pSubset         ( nullptr )
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------
//      Rs[RXgN^ł.
//-------------------------------------------------------------------------------------------
Mesh::Mesh( const Mesh& mesh )
: m_VertexCount     ( mesh.m_VertexCount )
, m_IndexCount      ( mesh.m_IndexCount )
, m_MaterialCount   ( mesh.m_MaterialCount )
, m_SubsetCount     ( mesh.m_SubsetCount )
{
    // _̃m.
    m_pVertex = new Mesh::Vertex[ m_VertexCount ];
    assert( m_pVertex != nullptr );

    // _CfbNX̃m.
    m_pIndex = new Mesh::Index[ m_IndexCount ];
    assert( m_pIndex != nullptr );

    // }eÃm.
    m_pMaterial = new Mesh::Material[ m_MaterialCount ];
    assert( m_pMaterial != nullptr );

    // TuZbg̃m.
    m_pSubset = new Mesh::Subset[ m_SubsetCount ];
    assert( m_pSubset != nullptr );

    // _f[^Rs[.
    memcpy( m_pVertex, mesh.m_pVertex, sizeof( Mesh::Vertex ) * mesh.m_VertexCount );

    // _CfbNXf[^Rs[.
    memcpy( m_pIndex, mesh.m_pIndex, sizeof( Mesh::Index ) * mesh.m_IndexCount );

    // }eAf[^Rs[.
    memcpy( m_pMaterial, mesh.m_pMaterial, sizeof( Mesh::Material ) * mesh.m_MaterialCount );

    // TuZbgf[^Rs[.
    memcpy( m_pSubset, mesh.m_pSubset, sizeof( Mesh::Subset ) * mesh.m_SubsetCount );
}

//-------------------------------------------------------------------------------------------
//      fXgN^ł.
//-------------------------------------------------------------------------------------------
Mesh::~Mesh()
{
    // .
    Release(); 
}

//-------------------------------------------------------------------------------------------
//      ܂.
//-------------------------------------------------------------------------------------------
void Mesh::Release()
{
    // _f[^̃.
    if ( m_pVertex )
    {
        delete [] m_pVertex;
        m_pVertex = nullptr;
    }

    // _CfbNXf[^̃.
    if ( m_pIndex )
    {
        delete [] m_pIndex;
        m_pIndex = nullptr;
    }

    // }eAf[^̃.
    if ( m_pMaterial )
    {
        delete [] m_pMaterial;
        m_pMaterial = nullptr;
    }

    // TuZbgf[^̃.
    if ( m_pSubset )
    {
        delete [] m_pSubset;
        m_pSubset = nullptr;
    }

    // JEgZbg.
    m_VertexCount   = 0;
    m_IndexCount    = 0;
    m_MaterialCount = 0;
    m_SubsetCount   = 0;
}

//--------------------------------------------------------------------------------------------
//      Zqł.
//--------------------------------------------------------------------------------------------
Mesh& Mesh::operator = ( const Mesh& mesh )
{
    // Ă.
    Release();

    // JEgݒ.
    m_VertexCount   = mesh.m_VertexCount;
    m_IndexCount    = mesh.m_IndexCount;
    m_MaterialCount = mesh.m_MaterialCount;
    m_SubsetCount   = mesh.m_SubsetCount;

    // _f[^̃m.
    m_pVertex = new Mesh::Vertex[ m_VertexCount ];
    assert( m_pVertex != nullptr );

    // _CfbNXf[^̃m.
    m_pIndex = new Mesh::Index[ m_IndexCount ];
    assert( m_pIndex != nullptr );

    // }eAf[^̃m.
    m_pMaterial = new Mesh::Material[ m_MaterialCount ];
    assert( m_pMaterial != nullptr );

    // TuZbgf[^̃m.
    m_pSubset = new Mesh::Subset[ m_SubsetCount ];
    assert( m_pSubset != nullptr );

    // _f[^Rs[.
    memcpy( m_pVertex, mesh.m_pVertex, sizeof( Mesh::Vertex ) * mesh.m_VertexCount );

    // _CfbNXf[^Rs[.
    memcpy( m_pIndex, mesh.m_pIndex, sizeof( Mesh::Index ) * mesh.m_IndexCount );

    // }eAf[^Rs[.
    memcpy( m_pMaterial, mesh.m_pMaterial, sizeof( Mesh::Material ) * mesh.m_MaterialCount );

    // TuZbgf[^Rs[.
    memcpy( m_pSubset, mesh.m_pSubset, sizeof( Mesh::Subset ) * mesh.m_SubsetCount );

    return (*this);
}

//-------------------------------------------------------------------------------------------
//      _擾܂.
//-------------------------------------------------------------------------------------------
asdx::UINT Mesh::GetVertexCount() const
{ return m_VertexCount; }

//-------------------------------------------------------------------------------------------
//      _CfbNX擾܂.
//-------------------------------------------------------------------------------------------
asdx::UINT Mesh::GetIndexCount() const
{ return m_IndexCount; }

//-------------------------------------------------------------------------------------------
//      }eA擾܂.
//-------------------------------------------------------------------------------------------
asdx::UINT Mesh::GetMaterialCount() const
{ return m_MaterialCount; }

//-------------------------------------------------------------------------------------------
//      TuZbg擾܂.
//-------------------------------------------------------------------------------------------
asdx::UINT Mesh::GetSubsetCount() const
{ return m_SubsetCount; }

//-------------------------------------------------------------------------------------------
//      w肳ꂽ_f[^擾܂.
//-------------------------------------------------------------------------------------------
asdx::Mesh::Vertex Mesh::GetVertex( const asdx::UINT idx ) const
{
    assert( idx < m_VertexCount );
    return m_pVertex[ idx ];
}

//-------------------------------------------------------------------------------------------
//      w肳ꂽ_CfbNX擾܂.
//-------------------------------------------------------------------------------------------
asdx::Mesh::Index  Mesh::GetIndex( const asdx::UINT idx ) const
{
    assert( idx < m_IndexCount );
    return m_pIndex[ idx ];
}

//-------------------------------------------------------------------------------------------
//      w肳ꂽ}eA擾܂.
//-------------------------------------------------------------------------------------------
asdx::Mesh::Material Mesh::GetMatrial( const asdx::UINT idx ) const
{
    assert( idx < m_MaterialCount );
    return m_pMaterial[ idx ];
}

//-------------------------------------------------------------------------------------------
//      w肳ꂽTuZbg擾܂.
//-------------------------------------------------------------------------------------------
asdx::Mesh::Subset Mesh::GetSubset( const asdx::UINT idx ) const
{
    assert( idx < m_SubsetCount );
    return m_pSubset[ idx ];
}

//-------------------------------------------------------------------------------------------
//      _f[^擾܂.
//-------------------------------------------------------------------------------------------
const asdx::Mesh::Vertex* Mesh::GetVertices() const
{ return m_pVertex; }

//-------------------------------------------------------------------------------------------
//      _CfbNX擾܂.
//-------------------------------------------------------------------------------------------
const asdx::Mesh::Index* Mesh::GetIndices() const
{ return m_pIndex; }

//-------------------------------------------------------------------------------------------
//      }eA擾܂.
//-------------------------------------------------------------------------------------------
const asdx::Mesh::Material* Mesh::GetMaterials() const
{ return m_pMaterial; }

//-------------------------------------------------------------------------------------------
//      TuZbg擾܂.
//-------------------------------------------------------------------------------------------
const asdx::Mesh::Subset* Mesh::GetSubsets() const
{ return m_pSubset; }

//-------------------------------------------------------------------------------------------
//      t@Cf[^[h܂.
//-------------------------------------------------------------------------------------------
bool Mesh::LoadFromFile( const char* filename )
{
    FILE* pFile;

    // t@CJ.
    errno_t err = fopen_s( &pFile, filename, "rb" );
    if ( err != 0 )
    {
        printf_s( "Error : LoadFromFile() Failed. filename = %s\n", filename );
        return false;
    }

    // t@Cwb_[ǂݎ.
    MSH_FILE_HEADER header;
    fread( &header, sizeof( MSH_FILE_HEADER ), 1, pFile );

    // }WbN`FbN.
    if ( ( header.Magic[ 0 ] != 'M' )
      || ( header.Magic[ 1 ] != 'S' )
      || ( header.Magic[ 2 ] != 'H' )
      || ( header.Magic[ 3 ] != '\0' ) )
    {
        printf_s( "Error : Invalid File.\n");
        fclose( pFile );
        return false;
    }

    // t@Co[W`FbN.
    if ( header.Version != MSH_CURRENT_VERSION )
    {
        printf_s( "Error : Invalid File Version.\n" );
        fclose( pFile );
        return false;
    }

    // f[^wb_[TCY`FbN.
    if ( header.DataHeaderSize != sizeof( MSH_DATA_HEADER ) )
    {
        printf_s( "Error : Data Header Size Not Matched.\n" );
        fclose( pFile );
        return false;
    }

    // _f[^TCY`FbN.
    if ( header.DataHeader.VertexStructureSize != sizeof( Mesh::Vertex ) )
    {
        printf_s( "Error : Vertex Structure Size Not Matched. expect size = %d, value = %d\n", sizeof( Mesh::Vertex ), header.DataHeader.VertexStructureSize );
        fclose( pFile );
        return false;
    }

    // _CfbNXf[^TCY`FbN.
    if ( header.DataHeader.IndexStructureSize != sizeof( asdx::UINT ) )
    {
        printf_s( "Error : Index Structure Size Not Matched. expect size = %d, value = %d\n", sizeof( asdx::UINT ), header.DataHeader.IndexStructureSize );
        fclose( pFile );
        return false;
    }

    // }eAf[^TCY`FbN.
    if ( header.DataHeader.MaterialStructureSize != sizeof( Mesh::Material ) )
    {
        printf_s( "Error : Material Structure Size Not Matched. expect size = %d, value = %d\n", sizeof( Mesh::Material ), header.DataHeader.MaterialStructureSize );
        fclose( pFile );
        return false;
    }

    // TuZbgf[^TCY`FbN.
    if ( header.DataHeader.SubsetStructureSize != sizeof( Mesh::Subset ) )
    {
        printf_s( "Error : Subset Structure Size Not Matched. expect size = %d, value = %d\n", sizeof( Mesh::Subset ), header.DataHeader.SubsetStructureSize );
        fclose( pFile );
        return false;
    }

    // JEgݒ.
    m_VertexCount   = header.DataHeader.NumVertices;
    m_IndexCount    = header.DataHeader.NumIndices;
    m_MaterialCount = header.DataHeader.NumMaterials;
    m_SubsetCount   = header.DataHeader.NumSubsets;

#if 0
    //// fobOO.
    //printf_s( "VertexCount   : %d\n", m_VertexCount );
    //printf_s( "IndexCount    : %d\n", m_IndexCount );
    //printf_s( "MaterialCount : %d\n", m_MaterialCount );
    //printf_s( "SubsetCount   : %d\n", m_SubsetCount );
#endif

    // _f[^̃m.
    m_pVertex = new Vertex[ m_VertexCount ];
    if ( m_pVertex == nullptr )
    {
        //@G[Oo.
        printf_s( "Error : Memory Allocate Failed.\n" );

        // t@C.
        fclose( pFile );

        // ƃJEgZbg.
        Release();

        // ُI.
        return false;
    }

    // _CfbNXf[^̃m.
    m_pIndex = new Index[ m_IndexCount ];
    if ( m_pIndex == nullptr )
    {
        // G[Oo.
        printf_s( "Error : Memory Allocate Failed.\n" );

        // t@C.
        fclose( pFile );

        // ƃJEgZbg.
        Release();

        // ُI.
        return false;
    }

    // }eAf[^̃m.
    m_pMaterial = new Material[ m_MaterialCount ];
    if ( m_pMaterial == nullptr )
    {
        // G[Oo.
        printf_s( "Error : Memory Allocate Failed.\n" );

        // t@C.
        fclose( pFile );

        // ƃJEgZbg.
        Release();

        // ُI.
        return false;
    }

    // TuZbgf[^̃m.
    m_pSubset = new Subset[ m_SubsetCount ];
    if ( m_pSubset == nullptr )
    {
        // G[Oo.
        printf_s( "Error : Memory Allocate Failed.\n" );

        // t@C.
        fclose( pFile );

        // ƃJEgZbg.
        Release();

        // ُI.
        return false;
    }

    // f[^Cɓǂݍ.
    fread( m_pVertex,   sizeof( Vertex ),   m_VertexCount,   pFile );
    fread( m_pIndex,    sizeof( Index ),    m_IndexCount,    pFile );
    fread( m_pMaterial, sizeof( Material ), m_MaterialCount, pFile );
    fread( m_pSubset,   sizeof( Subset ),   m_SubsetCount,   pFile );

#if 0
    //for( asdx::UINT i=0; i<m_VertexCount; ++i )
    //{
    //    printf_s( "Vertex[ %d ] :\n", i );
    //    printf_s( "Position ( %f, %f, %f )\n", m_pVertex[ i ].Position.x, m_pVertex[ i ].Position.y, m_pVertex[ i ].Position.z );
    //    printf_s( "Normal   ( %f, %f, %f )\n", m_pVertex[ i ].Normal.x, m_pVertex[ i ].Normal.y, m_pVertex[ i ].Normal.z );
    //    printf_s( "TexCoord ( %f, %f )\n", m_pVertex[ i ].TexCoord.x, m_pVertex[ i ].TexCoord.y );
    //    printf_s( "\n" );
    //}
    //for( asdx::UINT i=0; i<m_IndexCount; ++i )
    //{
    //    printf_s( "Index[ %d ] = %u\n", i, m_pIndex[ i ] ); 
    //}
    //for( asdx::UINT i=0; i<m_MaterialCount; ++i )
    //{
    //    printf_s( "Material[ %d ] :\n", i );
    //    printf_s( "Ambient ( %f, %f, %f )\n", m_pMaterial[ i ].Ambient.x, m_pMaterial[ i ].Ambient.y, m_pMaterial[ i ].Ambient.z );
    //    printf_s( "Diffuse ( %f, %f, %f )\n", m_pMaterial[ i ].Diffuse.x, m_pMaterial[ i ].Diffuse.y, m_pMaterial[ i ].Diffuse.z );
    //    printf_s( "Specular( %f, %f, %f )\n", m_pMaterial[ i ].Specular.x, m_pMaterial[ i ].Specular.y, m_pMaterial[ i ].Specular.z );
    //    printf_s( "Emissive( %f, %f, %f )\n", m_pMaterial[ i ].Emissive.x, m_pMaterial[ i ].Emissive.y, m_pMaterial[ i ].Emissive.z );
    //    printf_s( "Power   ( %f )\n", m_pMaterial[ i ].Power );
    //    printf_s( "Alpha   ( %f )\n", m_pMaterial[ i ].Alpha );
    //    printf_s( "AmbientMap      [%s]\n", m_pMaterial[ i ].AmbientMap );
    //    printf_s( "DiffuseMap      [%s]\n", m_pMaterial[ i ].DiffuseMap );
    //    printf_s( "SpecularMap     [%s]\n", m_pMaterial[ i ].SpecularMap );
    //    printf_s( "BumpMap         [%s]\n", m_pMaterial[ i ].BumpMap );
    //    printf_s( "DisplacementMap [%s]\n", m_pMaterial[ i ].DisplacementMap );
    //    printf_s( "\n" );
    //}
    //for( asdx::UINT i=0; i<m_SubsetCount; ++i )
    //{
    //    printf_s( "Subset[ %d ] :\n", i );
    //    printf_s( "IndexOffset ( %d )\n", m_pSubset[ i ].IndexOffset );
    //    printf_s( "IndexCount  ( %d )\n", m_pSubset[ i ].IndexCount );
    //    printf_s( "MaterialID  ( %d )\n", m_pSubset[ i ].MaterialID );
    //    printf_s( "\n" );
    //}
#endif

    // t@C.
    fclose( pFile );

    // I.
    return true;
}

} // namespace asdx

